import React, { useState, useMemo } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Wallet } from '../types';
import WalletModal from './WalletModal';
import TransferModal from './TransferModal';
import { PlusIcon, PencilSquareIcon, TrashIcon, ArrowsRightLeftIcon, BanknotesIcon, BuildingLibraryIcon, CreditCardIcon, CircleStackIcon, Bars3Icon } from './icons/Icons';

type DropIndicator = 'top' | 'bottom' | null;

const WalletTypeIcon = ({ type, className }: { type: Wallet['type'], className?: string }) => {
    switch (type) {
        case 'cash':
            return <BanknotesIcon className={className} />;
        case 'bank':
            return <BuildingLibraryIcon className={className} />;
        case 'credit':
            return <CreditCardIcon className={className} />;
        case 'crypto':
            return <CircleStackIcon className={className} />;
        default:
            return <CreditCardIcon className={className} />;
    }
};

const Wallets: React.FC = () => {
    const { wallets, deleteWallet, getWalletBalance, transactions, updateAllWallets, updateWallet } = useData();
    const { t, language } = useLanguage();
    const [isWalletModalOpen, setIsWalletModalOpen] = useState(false);
    const [isTransferModalOpen, setIsTransferModalOpen] = useState(false);
    const [editingWallet, setEditingWallet] = useState<Wallet | null>(null);

    const [draggedId, setDraggedId] = useState<string | null>(null);
    const [overId, setOverId] = useState<string | null>(null);
    const [dropIndicator, setDropIndicator] = useState<DropIndicator>(null);

    const handleAddClick = () => {
        setEditingWallet(null);
        setIsWalletModalOpen(true);
    };

    const handleEditClick = (wallet: Wallet) => {
        setEditingWallet(wallet);
        setIsWalletModalOpen(true);
    };

    const handleDeleteClick = (id: string) => {
        if (window.confirm(t('confirmDeleteWallet'))) {
            deleteWallet(id);
        }
    };
    
    const handleToggleShowOnDashboard = (wallet: Wallet) => {
        const currentlyShown = wallets.filter(w => w.showOnDashboard).length;
        if (!wallet.showOnDashboard && currentlyShown >= 4) {
            alert(t('maxWalletsOnDashboardError'));
            return;
        }
        updateWallet({ ...wallet, showOnDashboard: !wallet.showOnDashboard });
    };

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat(language === 'vi' ? 'vi-VN' : 'en-US', {
            style: 'currency',
            currency: language === 'vi' ? 'VND' : 'USD'
        }).format(amount);
    }

    const walletsForRender = useMemo(() => {
        const getSpent = (walletId: string, childIds: string[]) => {
            const idsToSum = [walletId, ...childIds];
            return transactions
                .filter(t => idsToSum.includes(t.walletId) && t.type === 'expense' && t.categoryId !== 'cat-transfer')
                .reduce((sum, t) => sum + t.amount, 0);
        };

        const parentWallets: Wallet[] = [];
        const childWallets = new Map<string, Wallet[]>();

        wallets.forEach(w => {
            if (w.parentWalletId) {
                if (!childWallets.has(w.parentWalletId)) {
                    childWallets.set(w.parentWalletId, []);
                }
                childWallets.get(w.parentWalletId)!.push(w);
            } else {
                parentWallets.push(w);
            }
        });

        parentWallets.sort((a, b) => a.order - b.order);

        const result: (Wallet & { isChild: boolean; spent?: number; limit?: number; progress?: number })[] = [];
        
        parentWallets.forEach(parent => {
            let limit = 0;
            let spent = 0;
            let progress = 0;
            const children = childWallets.get(parent.id) || [];
            
            if (parent.type === 'credit' && parent.creditLimit && parent.creditLimit > 0) {
                const childIds = children.map(c => c.id);
                limit = parent.creditLimit;
                spent = getSpent(parent.id, childIds);
                progress = (spent / limit) * 100;
            }
            
            result.push({ ...parent, isChild: false, spent, limit, progress });
            
            children.forEach(child => {
                result.push({ ...child, isChild: true });
            });
        });

        return result;
    }, [wallets, transactions]);
    
    const handleDragStart = (e: React.DragEvent, wallet: Wallet) => {
        e.dataTransfer.setData('walletId', wallet.id);
        e.dataTransfer.effectAllowed = 'move';
        setTimeout(() => setDraggedId(wallet.id), 0);
    };

    const handleDragEnd = () => {
        setDraggedId(null);
        setOverId(null);
        setDropIndicator(null);
    };

    const handleDragOver = (e: React.DragEvent, wallet: Wallet) => {
        e.preventDefault();
        if (!draggedId || wallet.id === draggedId || wallet.parentWalletId) {
            setOverId(null);
            setDropIndicator(null);
            return;
        }
        setOverId(wallet.id);

        const rect = e.currentTarget.getBoundingClientRect();
        const relativeY = e.clientY - rect.top;
        const height = rect.height;

        if (relativeY < height / 2) setDropIndicator('top');
        else setDropIndicator('bottom');
    };

    const handleDrop = (e: React.DragEvent, dropTarget: Wallet) => {
        e.preventDefault();
        const draggedWalletId = e.dataTransfer.getData('walletId');

        if (!draggedWalletId || !dropIndicator || draggedWalletId === dropTarget.id) {
            handleDragEnd();
            return;
        }

        const updatedWallets = JSON.parse(JSON.stringify(wallets)) as Wallet[];
        const draggedItem = updatedWallets.find(w => w.id === draggedWalletId)!;

        draggedItem.order = dropTarget.order + (dropIndicator === 'bottom' ? 0.5 : -0.5);
        
        const parentWallets = updatedWallets
            .filter(w => !w.parentWalletId)
            .sort((a, b) => a.order - b.order);
            
        parentWallets.forEach((wallet, index) => {
            const itemToUpdate = updatedWallets.find(w => w.id === wallet.id)!;
            itemToUpdate.order = index;
        });

        updateAllWallets(updatedWallets);
        handleDragEnd();
    };

    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <h1 className="text-3xl font-bold text-on-surface">{t('wallets')}</h1>
                <div className="flex gap-4">
                    <button 
                        onClick={() => setIsTransferModalOpen(true)}
                        className="flex items-center gap-2 bg-secondary text-white px-4 py-2 rounded-lg hover:bg-green-600 transition-colors"
                    >
                        <ArrowsRightLeftIcon className="w-5 h-5" />
                        <span>{t('transferFunds')}</span>
                    </button>
                    <button 
                        onClick={handleAddClick} 
                        className="flex items-center gap-2 bg-primary text-white px-4 py-2 rounded-lg hover:bg-primary-dark transition-colors"
                    >
                        <PlusIcon className="w-5 h-5" />
                        <span>{t('addWallet')}</span>
                    </button>
                </div>
            </div>

            <div className="bg-surface rounded-xl shadow-sm overflow-x-auto">
                <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                        <tr>
                            <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-on-surface-secondary uppercase tracking-wider">{t('walletName')}</th>
                            <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-on-surface-secondary uppercase tracking-wider">{t('walletType')}</th>
                            <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-on-surface-secondary uppercase tracking-wider">{t('limitUsage')}</th>
                            <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-on-surface-secondary uppercase tracking-wider">{t('currentBalance')}</th>
                            <th scope="col" className="px-6 py-3 text-center text-xs font-medium text-on-surface-secondary uppercase tracking-wider">{t('dashboardDisplay')}</th>
                            <th scope="col" className="px-6 py-3 text-center text-xs font-medium text-on-surface-secondary uppercase tracking-wider">{t('actions')}</th>
                        </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                        {walletsForRender.map(wallet => {
                            const parentInfo = wallet.isChild ? walletsForRender.find(w => w.id === wallet.parentWalletId) : null;
                            const limit = wallet.isChild ? parentInfo?.limit : wallet.limit;
                            const spent = wallet.isChild ? parentInfo?.spent : wallet.spent;
                            const progress = wallet.isChild ? parentInfo?.progress : wallet.progress;

                            return (
                                <tr 
                                    key={wallet.id}
                                    className={`${wallet.isChild ? 'bg-gray-50/50' : ''} ${draggedId === wallet.id ? 'opacity-30' : ''}`}
                                    draggable={!wallet.isChild}
                                    onDragStart={!wallet.isChild ? (e) => handleDragStart(e, wallet) : undefined}
                                    onDragEnd={!wallet.isChild ? handleDragEnd : undefined}
                                    onDragOver={!wallet.isChild ? (e) => handleDragOver(e, wallet) : undefined}
                                    onDrop={!wallet.isChild ? (e) => handleDrop(e, wallet) : undefined}
                                    onDragLeave={() => { setOverId(null); setDropIndicator(null); }}
                                >
                                    <td className={`px-6 py-4 whitespace-nowrap text-sm font-medium text-on-surface relative ${wallet.isChild ? 'pl-10' : ''}`}>
                                        {overId === wallet.id && dropIndicator === 'top' && <div className="absolute -top-0.5 left-0 w-full h-1 bg-primary z-10 rounded-full" />}
                                        <div className="flex items-center gap-2">
                                            {!wallet.isChild ? (
                                                <span className="cursor-move text-gray-400 hover:text-gray-800">
                                                    <Bars3Icon className="w-5 h-5" />
                                                </span>
                                            ) : (
                                                <span className="mr-2 text-gray-400">└</span>
                                            )}
                                            <span>{wallet.name}</span>
                                        </div>
                                        {overId === wallet.id && dropIndicator === 'bottom' && <div className="absolute -bottom-0.5 left-0 w-full h-1 bg-primary z-10 rounded-full" />}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-on-surface-secondary">
                                        <div className="flex items-center gap-2">
                                            <WalletTypeIcon type={wallet.type} className="w-5 h-5" />
                                            <span>{t(wallet.type)}</span>
                                        </div>
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm">
                                        {wallet.type === 'credit' && limit && limit > 0 && (
                                            <div className="w-48">
                                                <div className="flex justify-between text-xs mb-1 text-on-surface-secondary">
                                                    <span>{formatCurrency(spent ?? 0)}</span>
                                                    <span className="font-semibold text-on-surface">{formatCurrency(limit)}</span>
                                                </div>
                                                <div className="w-full bg-gray-200 rounded-full h-2">
                                                    <div 
                                                        className="bg-orange-500 h-2 rounded-full"
                                                        style={{ width: `${Math.min(progress ?? 0, 100)}%` }}
                                                    ></div>
                                                </div>
                                            </div>
                                        )}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-sm font-semibold text-right text-on-surface-secondary">
                                        {formatCurrency(getWalletBalance(wallet.id))}
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-center">
                                        <button
                                            onClick={() => handleToggleShowOnDashboard(wallet)}
                                            className={`relative inline-flex h-6 w-11 flex-shrink-0 cursor-pointer rounded-full border-2 border-transparent transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-primary focus:ring-offset-2 ${wallet.showOnDashboard ? 'bg-primary' : 'bg-gray-200'}`}
                                            role="switch"
                                            aria-checked={wallet.showOnDashboard}
                                        >
                                            <span
                                                aria-hidden="true"
                                                className={`inline-block h-5 w-5 transform rounded-full bg-white shadow ring-0 transition duration-200 ease-in-out ${wallet.showOnDashboard ? 'translate-x-5' : 'translate-x-0'}`}
                                            />
                                        </button>
                                    </td>
                                    <td className="px-6 py-4 whitespace-nowrap text-center text-sm font-medium">
                                        <button onClick={() => handleEditClick(wallet)} className="text-primary hover:text-primary-dark p-1" title={t('editWallet')}>
                                            <PencilSquareIcon className="w-5 h-5"/>
                                        </button>
                                        <button onClick={() => handleDeleteClick(wallet.id)} className="text-red-600 hover:text-red-800 p-1 ml-2" title={t('delete')}>
                                            <TrashIcon className="w-5 h-5"/>
                                        </button>
                                    </td>
                                </tr>
                            );
                        })}
                    </tbody>
                </table>
                 {wallets.length === 0 && <p className="p-6 text-center text-on-surface-secondary">No wallets found.</p>}
            </div>
            
            <WalletModal
                isOpen={isWalletModalOpen}
                onClose={() => setIsWalletModalOpen(false)}
                initialData={editingWallet}
            />
             <TransferModal
                isOpen={isTransferModalOpen}
                onClose={() => setIsTransferModalOpen(false)}
            />
        </div>
    );
};

export default Wallets;