
import React, { useState, useEffect, FormEvent } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Wallet } from '../types';
import { XMarkIcon } from './icons/Icons';

interface WalletModalProps {
    isOpen: boolean;
    onClose: () => void;
    initialData?: Wallet | null;
}

const WalletModal: React.FC<WalletModalProps> = ({ isOpen, onClose, initialData }) => {
    const { addWallet, updateWallet, wallets } = useData();
    const { t, formatCurrencyInput, parseCurrencyInput } = useLanguage();
    
    const [name, setName] = useState('');
    const [initialBalance, setInitialBalance] = useState('');
    const [type, setType] = useState<Wallet['type']>('bank');
    const [creditLimit, setCreditLimit] = useState('');
    const [parentWalletId, setParentWalletId] = useState<string | null>(null);
    const [showOnDashboard, setShowOnDashboard] = useState(false);
    const [error, setError] = useState('');

    useEffect(() => {
        if (isOpen) {
            if (initialData) {
                setName(initialData.name);
                setInitialBalance(formatCurrencyInput(String(initialData.initialBalance)));
                setType(initialData.type);
                setCreditLimit(formatCurrencyInput(String(initialData.creditLimit || '')));
                setParentWalletId(initialData.parentWalletId || null);
                setShowOnDashboard(initialData.showOnDashboard || false);
            } else {
                setName('');
                setInitialBalance('0');
                setType('bank');
                setCreditLimit('');
                setParentWalletId(null);
                const shownWallets = wallets.filter(w => w.showOnDashboard).length;
                setShowOnDashboard(shownWallets < 2);
            }
            setError('');
        }
    }, [initialData, isOpen, formatCurrencyInput, wallets]);

    const handleToggleShowOnDashboard = (e: React.ChangeEvent<HTMLInputElement>) => {
        const isChecked = e.target.checked;
        const currentlyShown = wallets.filter(w => w.showOnDashboard).length;
        
        if (isChecked && currentlyShown >= 4 && !(initialData && initialData.showOnDashboard)) {
            alert(t('maxWalletsOnDashboardError'));
            return;
        }
        setShowOnDashboard(isChecked);
    };

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        const numBalance = parseCurrencyInput(initialBalance);
        const numCreditLimit = parseCurrencyInput(creditLimit);
        
        if (!name) {
            setError(t('allFieldsRequired'));
            return;
        }
        
        if (type === 'credit' && !parentWalletId && (isNaN(numCreditLimit) || numCreditLimit <= 0)) {
             setError('Please enter a valid credit limit for the main card.');
             return;
        }
        setError('');

        const walletData = {
            name,
            initialBalance: numBalance,
            type,
            creditLimit: type === 'credit' && !parentWalletId ? numCreditLimit : undefined,
            parentWalletId: type === 'credit' ? parentWalletId : null,
            showOnDashboard,
        };
        
        if (initialData) {
            // FIX: The 'order' property is required by the Wallet type. Pass it from initialData.
            updateWallet({ ...walletData, id: initialData.id, order: initialData.order });
        } else {
            addWallet(walletData);
        }
        onClose();
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center z-50" aria-modal="true" role="dialog">
            <div className="bg-surface rounded-xl shadow-2xl w-full max-w-md m-4">
                <form onSubmit={handleSubmit}>
                    <div className="p-6 border-b border-gray-200 flex justify-between items-center">
                        <h2 className="text-xl font-bold">{initialData ? t('editWallet') : t('addWallet')}</h2>
                        <button type="button" onClick={onClose} className="text-on-surface-secondary hover:text-on-surface">
                            <XMarkIcon className="w-6 h-6" />
                        </button>
                    </div>

                    <div className="p-6 space-y-4 max-h-[70vh] overflow-y-auto">
                        <div>
                            <label htmlFor="name" className="block text-sm font-medium text-on-surface-secondary">{t('walletName')}</label>
                            <input type="text" id="name" value={name} onChange={e => setName(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                        </div>
                        <div className="grid grid-cols-2 gap-4">
                            <div>
                                <label htmlFor="initialBalance" className="block text-sm font-medium text-on-surface-secondary">{t('initialBalance')}</label>
                                <input 
                                    type="text" 
                                    inputMode="numeric" 
                                    id="initialBalance" 
                                    value={initialBalance} 
                                    onChange={e => setInitialBalance(formatCurrencyInput(e.target.value))} 
                                    placeholder="0" 
                                    className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                                />
                            </div>
                            <div>
                                <label htmlFor="walletType" className="block text-sm font-medium text-on-surface-secondary">{t('walletType')}</label>
                                <select id="walletType" value={type} onChange={e => setType(e.target.value as Wallet['type'])} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                    <option value="cash">{t('cash')}</option>
                                    <option value="bank">{t('bank')}</option>
                                    <option value="credit">{t('credit')}</option>
                                    <option value="crypto">{t('crypto')}</option>
                                </select>
                            </div>
                        </div>

                        {type === 'credit' && (
                            <div className='p-4 bg-primary/5 rounded-lg space-y-4'>
                                <h3 className='font-semibold text-primary'>{t('credit')}</h3>
                                <div>
                                    <label htmlFor="parentWallet" className="block text-sm font-medium text-on-surface-secondary">{t('parentWallet')}</label>
                                    <select id="parentWallet" value={parentWalletId || ''} onChange={e => setParentWalletId(e.target.value || null)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                        <option value="">{t('selectParentWallet')}</option>
                                        {wallets
                                            .filter(w => w.type === 'credit' && !w.parentWalletId && w.id !== initialData?.id)
                                            .map(w => <option key={w.id} value={w.id}>{w.name}</option>)
                                        }
                                    </select>
                                </div>
                                <div>
                                    <label htmlFor="creditLimit" className="block text-sm font-medium text-on-surface-secondary">{t('creditLimit')}</label>
                                    <input 
                                        type="text" 
                                        inputMode="numeric"
                                        id="creditLimit" 
                                        value={creditLimit} 
                                        onChange={e => setCreditLimit(formatCurrencyInput(e.target.value))}
                                        disabled={!!parentWalletId}
                                        placeholder={parentWalletId ? 'Sử dụng hạn mức thẻ chính' : '50.000.000'}
                                        className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface disabled:bg-gray-100"
                                    />
                                </div>
                            </div>
                        )}
                        
                        <div className="flex items-center pt-2">
                            <input
                                type="checkbox"
                                id="showOnDashboard"
                                checked={showOnDashboard}
                                onChange={handleToggleShowOnDashboard}
                                className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary"
                            />
                            <label htmlFor="showOnDashboard" className="ml-2 block text-sm text-on-surface">
                                {t('showOnDashboard')}
                            </label>
                        </div>

                        {error && <p className="text-sm text-red-600">{error}</p>}
                    </div>

                    <div className="p-6 bg-gray-50 rounded-b-xl flex justify-end items-center gap-4">
                        <button type="button" onClick={onClose} className="px-4 py-2 text-sm font-semibold text-on-surface-secondary bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                            {t('cancel')}
                        </button>
                        <button type="submit" className="px-4 py-2 text-sm font-semibold text-white bg-primary rounded-lg hover:bg-primary-dark transition-colors">
                            {t('save')}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default WalletModal;