
import React, { useState, useEffect, FormEvent, useMemo } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { XMarkIcon } from './icons/Icons';

interface TransferModalProps {
    isOpen: boolean;
    onClose: () => void;
}

const TransferModal: React.FC<TransferModalProps> = ({ isOpen, onClose }) => {
    const { wallets, addTransfer } = useData();
    const { t, formatDate, parseDate, formatCurrencyInput, parseCurrencyInput } = useLanguage();
    
    const [fromWalletId, setFromWalletId] = useState('');
    const [toWalletId, setToWalletId] = useState('');
    const [amount, setAmount] = useState('');
    const [date, setDate] = useState(''); // YYYY-MM-DD
    const [displayDate, setDisplayDate] = useState(''); // DD/MM/YYYY
    const [description, setDescription] = useState('');
    const [error, setError] = useState('');

    const sortedWallets = useMemo(() => {
        return [...wallets].sort((a, b) => a.order - b.order);
    }, [wallets]);

    useEffect(() => {
        if (isOpen) {
            if (sortedWallets.length > 0) {
                setFromWalletId(sortedWallets[0].id);
                if (sortedWallets.length > 1) {
                    setToWalletId(sortedWallets[1].id);
                } else {
                    setToWalletId('');
                }
            }
            const today = new Date().toISOString().split('T')[0];
            setAmount('');
            setDescription('');
            setDate(today);
            setDisplayDate(formatDate(today));
            setError('');
        }
    }, [isOpen, sortedWallets, formatDate]);

    const handleFromWalletChange = (val: string) => {
        setFromWalletId(val);
        setError('');
        // Nếu chọn Từ ví trùng với Đến ví, tự động đảo sang ví khác
        if (val === toWalletId) {
            const otherWallet = sortedWallets.find(w => w.id !== val);
            if (otherWallet) setToWalletId(otherWallet.id);
        }
    };

    const handleToWalletChange = (val: string) => {
        setToWalletId(val);
        setError('');
        // Nếu chọn Đến ví trùng với Từ ví, tự động đảo sang ví khác
        if (val === fromWalletId) {
            const otherWallet = sortedWallets.find(w => w.id !== val);
            if (otherWallet) setFromWalletId(otherWallet.id);
        }
    };

    const handleDateChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const newDisplayDate = e.target.value;
        setDisplayDate(newDisplayDate);
        setError('');
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            setDate(newIsoDate);
        }
    };

    const handleDateBlur = (e: React.FocusEvent<HTMLInputElement>) => {
        const newDisplayDate = e.target.value;
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            setDate(newIsoDate);
            setDisplayDate(formatDate(newIsoDate));
        } else if (newDisplayDate === '') {
            setDate('');
        } else {
            setDisplayDate(formatDate(date));
        }
    };

    const handleAmountChange = (val: string) => {
        setAmount(formatCurrencyInput(val));
        setError('');
    };

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        setError('');
        const numAmount = parseCurrencyInput(amount);
        
        if (!fromWalletId || !toWalletId || !amount || !date) {
            setError(t('allFieldsRequired'));
            return;
        }
        if (fromWalletId === toWalletId) {
            setError(t('walletsCannotBeSame'));
            return;
        }
        if (isNaN(numAmount) || numAmount <= 0) {
            setError(t('validAmount'));
            return;
        }
        
        addTransfer({
            fromWalletId,
            toWalletId,
            amount: numAmount,
            date,
            description,
        });
        onClose();
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center z-50" aria-modal="true" role="dialog">
            <div className="bg-surface rounded-xl shadow-2xl w-full max-w-md m-4">
                <form onSubmit={handleSubmit}>
                    <div className="p-6 border-b border-gray-200 flex justify-between items-center">
                        <h2 className="text-xl font-bold">{t('transferFunds')}</h2>
                        <button type="button" onClick={onClose} className="text-on-surface-secondary hover:text-on-surface">
                            <XMarkIcon className="w-6 h-6" />
                        </button>
                    </div>

                    <div className="p-6 space-y-4">
                        <div className="grid grid-cols-2 gap-4">
                            <div>
                                <label htmlFor="fromWallet" className="block text-sm font-medium text-on-surface-secondary">{t('fromWallet')}</label>
                                <select id="fromWallet" value={fromWalletId} onChange={e => handleFromWalletChange(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                    {sortedWallets.map(w => (
                                        <option key={w.id} value={w.id}>{w.name}</option>
                                    ))}
                                </select>
                            </div>
                             <div>
                                <label htmlFor="toWallet" className="block text-sm font-medium text-on-surface-secondary">{t('toWallet')}</label>
                                <select id="toWallet" value={toWalletId} onChange={e => handleToWalletChange(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                   {sortedWallets.map(w => (
                                        <option key={w.id} value={w.id}>{w.name}</option>
                                    ))}
                                </select>
                            </div>
                        </div>
                        <div>
                            <label htmlFor="transferAmount" className="block text-sm font-medium text-on-surface-secondary">{t('amount')}</label>
                            <input 
                                type="text" 
                                inputMode="numeric" 
                                id="transferAmount" 
                                value={amount} 
                                onChange={e => handleAmountChange(e.target.value)} 
                                placeholder="0" 
                                className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                            />
                        </div>
                        <div>
                            <label htmlFor="transferDate" className="block text-sm font-medium text-on-surface-secondary">{t('date')}</label>
                            <input 
                                type="text" 
                                id="transferDate" 
                                value={displayDate} 
                                onChange={handleDateChange} 
                                onBlur={handleDateBlur}
                                placeholder="dd/mm/yyyy"
                                className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                            />
                        </div>
                        <div>
                            <label htmlFor="transferDescription" className="block text-sm font-medium text-on-surface-secondary">{t('description')} <span className="text-xs">({t('optional')})</span></label>
                            <input type="text" id="transferDescription" value={description} onChange={e => setDescription(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                        </div>
                        {error && <p className="text-sm text-red-600">{error}</p>}
                    </div>

                    <div className="p-6 bg-gray-50 rounded-b-xl flex justify-end items-center gap-4">
                        <button type="button" onClick={onClose} className="px-4 py-2 text-sm font-semibold text-on-surface-secondary bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                            {t('cancel')}
                        </button>
                        <button type="submit" className="px-4 py-2 text-sm font-semibold text-white bg-primary rounded-lg hover:bg-primary-dark transition-colors">
                            {t('transfer')}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default TransferModal;
