

import React, { useMemo, useRef, useState, useCallback } from 'react';
import { useLanguage } from '../context/LanguageContext';
import { useData } from '../context/DataContext';
import { Category } from '../types';
import CategoryModal from './CategoryModal';
import { PlusIcon, PencilSquareIcon, TrashIcon, Icon, Bars3Icon } from './icons/Icons';

type DropIndicator = 'top' | 'bottom' | 'middle' | null;
type CategoryType = 'income' | 'expense';

// FIX: Changed type alias to interface for better recursive type handling.
interface CategoryTreeNode extends Category {
    children: CategoryTreeNode[];
}

const Settings: React.FC = () => {
    const { language, setLanguage, t } = useLanguage();
    const { categories, backupData, restoreData, deleteCategory, updateAllCategories } = useData();
    const restoreInputRef = useRef<HTMLInputElement>(null);
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editingCategory, setEditingCategory] = useState<Category | null>(null);
    const [newCategoryType, setNewCategoryType] = useState<CategoryType>('expense');

    const [draggedId, setDraggedId] = useState<string | null>(null);
    const [overId, setOverId] = useState<string | null>(null);
    const [dropIndicator, setDropIndicator] = useState<DropIndicator>(null);
    
    const categoryMap = useMemo(() => {
        const map = new Map<string, Category>();
        categories.forEach(cat => map.set(cat.id, cat));
        return map;
    }, [categories]);

    const getCategoryType = useCallback((catId: string | null): CategoryType | 'system' => {
        if (!catId) return 'expense';
        let current = categoryMap.get(catId);
        if (!current) return 'expense';

        if (['cat-transfer', 'cat-debt'].includes(current.id)) return 'system';
        
        while(current?.parentId) {
            current = categoryMap.get(current.parentId);
        }

        return current?.id === 'cat-i1' ? 'income' : 'expense';
    }, [categoryMap]);

    const { expenseTree, incomeTree, systemCategoriesList } = useMemo(() => {
        const tree: CategoryTreeNode[] = [];
        const map = new Map<string, CategoryTreeNode>();
        categories.forEach(cat => map.set(cat.id, { ...cat, children: [] }));
        
        map.forEach(cat => {
            if (cat.parentId && map.has(cat.parentId)) {
                map.get(cat.parentId)!.children.push(cat);
            } else {
                tree.push(cat);
            }
        });

        const sortRecursive = (arr: CategoryTreeNode[]) => {
            arr.sort((a, b) => a.order - b.order);
            arr.forEach((item: CategoryTreeNode) => sortRecursive(item.children));
        };
        sortRecursive(tree);

        const incomeRoot = tree.find(c => c.id === 'cat-i1');
        const incomeTree: CategoryTreeNode[] = incomeRoot ? [incomeRoot] : [];
        const systemCategoriesList = tree.filter(c => ['cat-transfer', 'cat-debt'].includes(c.id));
        const expenseTree = tree.filter(c => c.id !== 'cat-i1' && !['cat-transfer', 'cat-debt'].includes(c.id));

        return { expenseTree, incomeTree, systemCategoriesList };
    }, [categories]);
    
    const handleBackup = () => {
        const dataStr = backupData();
        const dataUri = 'data:application/json;charset=utf-8,'+ encodeURIComponent(dataStr);
        
        const linkElement = document.createElement('a');
        linkElement.setAttribute('href', dataUri);
        linkElement.setAttribute('download', 'ai-finance-backup.json');
        linkElement.click();
    };

    const handleRestoreClick = () => {
        restoreInputRef.current?.click();
    };
    
    const handleAddCategory = (type: CategoryType) => {
        setNewCategoryType(type);
        setEditingCategory(null);
        setIsModalOpen(true);
    };

    const handleEditCategory = (category: Category) => {
        setEditingCategory(category);
        setIsModalOpen(true);
    };
    
    const handleDeleteCategory = (id: string) => {
        if(window.confirm(t('confirmDeleteCategory'))) {
            const result = deleteCategory(id);
            if (!result.success) {
                alert(`${t('categoryDeleteError')}: ${result.message}`);
            }
        }
    };

    const handleFileChange = (event: React.ChangeEvent<HTMLInputElement>) => {
        const file = event.target.files?.[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = (e) => {
            try {
                const text = e.target?.result;
                if (typeof text !== 'string') throw new Error("File is not readable");
                
                const data = JSON.parse(text);
                
                if (window.confirm(t('confirmRestore'))) {
                    const result = restoreData(data);
                    if (result.success) {
                        alert(t('restoreSuccess'));
                        window.location.reload();
                    } else {
                        alert(t('restoreError'));
                    }
                }
            } catch (error) {
                console.error("Restore error:", error);
                alert(t('restoreError'));
            } finally {
                if (restoreInputRef.current) {
                    restoreInputRef.current.value = '';
                }
            }
        };
        reader.readAsText(file);
    };

    // --- Drag and Drop Handlers ---
    const handleDragStart = (e: React.DragEvent, category: Category) => {
        e.dataTransfer.setData('categoryId', category.id);
        e.dataTransfer.effectAllowed = 'move';
        setTimeout(() => setDraggedId(category.id), 0);
    };

    const handleDragEnd = () => {
        setDraggedId(null);
        setOverId(null);
        setDropIndicator(null);
    };

    const handleDragOver = (e: React.DragEvent, category: Category) => {
        e.preventDefault();
        if (!draggedId || category.id === draggedId) return;

        const draggedType = getCategoryType(draggedId);
        const targetType = getCategoryType(category.id);
        if (draggedType !== targetType) {
            setDropIndicator(null);
            return;
        }

        // Prevent dropping a parent into its own child
        let current: Category | undefined = category;
        while(current) {
            if(current.id === draggedId) {
                setOverId(null);
                setDropIndicator(null);
                return;
            }
            current = categories.find(c => c.id === current!.parentId);
        }

        setOverId(category.id);

        const rect = e.currentTarget.getBoundingClientRect();
        const relativeY = e.clientY - rect.top;
        const height = rect.height;

        if (relativeY < height / 3) setDropIndicator('top');
        else if (relativeY > (height * 2) / 3) setDropIndicator('bottom');
        else setDropIndicator('middle');
    };

    const handleDrop = (e: React.DragEvent, dropTarget: Category) => {
        e.preventDefault();
        const draggedCategoryId = e.dataTransfer.getData('categoryId');
    
        // --- 1. Validations ---
        if (!draggedCategoryId || !dropIndicator || draggedCategoryId === dropTarget.id) {
            handleDragEnd();
            return;
        }
        if (getCategoryType(draggedCategoryId) !== getCategoryType(dropTarget.id)) {
            handleDragEnd();
            return;
        }
    
        // --- 2. Create mutable copy ---
        const updatedCategories = JSON.parse(JSON.stringify(categories)) as Category[];
        const draggedItem = updatedCategories.find(c => c.id === draggedCategoryId)!;
        const originalParentId = draggedItem.parentId;
        let newParentId: string | null | undefined;
    
        // --- 3. Determine new parent and perform specific validations ---
        if (dropIndicator === 'middle') { // Case A: Reparenting
            newParentId = dropTarget.id;
    
            // Validation: Prevent dropping a parent into its own child
            let current: Category | undefined = dropTarget;
            while (current) {
                if (current.id === draggedCategoryId) {
                    handleDragEnd();
                    return; // Invalid drop
                }
                current = updatedCategories.find(c => c.id === current!.parentId);
            }
            
            draggedItem.parentId = newParentId;
            // The order will be handled by the re-ordering logic at the end.
        } else { // Case B: Reordering
            newParentId = dropTarget.parentId;
            draggedItem.parentId = newParentId;
    
            // Temporarily change order to be adjacent to target, using fractional numbers
            draggedItem.order = dropTarget.order + (dropIndicator === 'bottom' ? 0.5 : -0.5);
        }
        
        // --- 4. Re-normalize ordering for all affected groups ---
        
        // Function to re-index a list of siblings based on their current order
        const normalizeOrder = (parentId: string | null | undefined) => {
            const siblings = updatedCategories
                .filter(c => c.parentId === parentId)
                .sort((a, b) => a.order - b.order); // Sort by potentially fractional order
            
            siblings.forEach((item, index) => {
                // Find the item in the main list and update its order to the new integer index
                const itemToUpdate = updatedCategories.find(c => c.id === item.id)!;
                itemToUpdate.order = index;
            });
        };
    
        // If parent changed, normalize the old parent group
        if (originalParentId !== newParentId) {
            normalizeOrder(originalParentId);
        }
        
        // Always normalize the new parent group
        normalizeOrder(newParentId);
        
        updateAllCategories(updatedCategories);
        handleDragEnd();
    };


    const renderCategory = (category: CategoryTreeNode, level: number) => {
        const isSystemCategory = ['cat-transfer', 'cat-debt', 'cat-i1'].includes(category.id);
        const isDragging = draggedId === category.id;
        const isOver = overId === category.id;
        
        return (
             <div key={category.id} className="relative">
                {isOver && dropIndicator === 'top' && <div className="absolute top-0 left-0 w-full h-1 bg-primary z-10 rounded-full" />}
                <div 
                    draggable={!isSystemCategory}
                    onDragStart={!isSystemCategory ? (e) => handleDragStart(e, category) : undefined}
                    onDragEnd={!isSystemCategory ? handleDragEnd : undefined}
                    onDragOver={!isSystemCategory ? (e) => handleDragOver(e, category) : undefined}
                    onDrop={!isSystemCategory ? (e) => handleDrop(e, category) : undefined}
                    onDragLeave={() => { setOverId(null); setDropIndicator(null); }}
                    className={`
                        group relative flex items-center justify-between transition-all
                        ${level > 0 ? 'border-t' : ''}
                        ${isDragging ? 'opacity-30' : 'opacity-100'}
                        ${isOver && dropIndicator === 'middle' ? 'bg-primary/10 ring-2 ring-primary rounded-md' : ''}
                    `}
                    style={{ marginLeft: `${level * 20}px` }}
                >
                    <div className="flex items-center gap-3 py-2 flex-grow">
                        {!isSystemCategory ? (
                            <div className="cursor-move text-gray-400 group-hover:text-gray-800">
                                <Bars3Icon className="w-5 h-5"/>
                            </div>
                        ) : <div className="w-5 h-5"/>}
                        <span className="w-7 h-7 rounded-full flex items-center justify-center flex-shrink-0" style={{ backgroundColor: category.color + '20' }}>
                            <Icon name={category.icon} className="w-4 h-4" style={{ color: category.color }} />
                        </span>
                        <span className="font-semibold">{category.name}</span>
                    </div>
                    {!['cat-transfer', 'cat-debt'].includes(category.id) && (
                        <div>
                             <button onClick={() => handleEditCategory(category)} className="p-1 text-primary hover:text-primary-dark">
                                <PencilSquareIcon className="w-5 h-5" />
                            </button>
                            <button onClick={() => handleDeleteCategory(category.id)} className="p-1 text-red-600 hover:text-red-800 ml-2" disabled={isSystemCategory}>
                                <TrashIcon className="w-5 h-5"/>
                            </button>
                        </div>
                    )}
                </div>
                 {isOver && dropIndicator === 'bottom' && <div className="absolute bottom-0 left-0 w-full h-1 bg-primary z-10 rounded-full" />}
                {category.children.length > 0 && category.children.map((child) => renderCategory(child, level + 1))}
            </div>
        );
    }
    
    const CategorySection: React.FC<{title: string, tree: CategoryTreeNode[], type: CategoryType, children?: React.ReactNode}> = ({ title, tree, type, children }) => (
         <div className="bg-surface p-6 rounded-xl shadow-sm">
            <div className="flex justify-between items-center mb-4">
                <h2 className="text-xl font-bold">{title}</h2>
                 <button 
                    onClick={() => handleAddCategory(type)}
                    className="flex items-center gap-2 bg-primary text-white px-3 py-1.5 text-sm rounded-lg hover:bg-primary-dark transition-colors"
                >
                    <PlusIcon className="w-4 h-4" />
                    <span>{t('addCategory')}</span>
                </button>
            </div>
            <div className="bg-gray-50 p-2 rounded-lg border divide-y">
                {tree.map(cat => renderCategory(cat, 0))}
                {tree.length === 0 && <p className='text-center text-sm text-gray-500 py-4'>No categories found.</p>}
            </div>
             {children}
        </div>
    )

    return (
        <div className="space-y-6 max-w-2xl mx-auto">
            <h1 className="text-3xl font-bold text-on-surface">{t('settings')}</h1>

            <div className="bg-surface p-6 rounded-xl shadow-sm">
                <h2 className="text-xl font-bold mb-4">{t('language')}</h2>
                <p className="text-on-surface-secondary mb-3">{t('languagePrompt')}</p>
                <div className="flex space-x-2">
                    <button
                        onClick={() => setLanguage('vi')}
                        className={`px-4 py-2 rounded-lg transition-colors ${
                            language === 'vi' ? 'bg-primary text-white' : 'bg-gray-200 hover:bg-gray-300'
                        }`}
                    >
                        Tiếng Việt
                    </button>
                    <button
                        onClick={() => setLanguage('en')}
                        className={`px-4 py-2 rounded-lg transition-colors ${
                            language === 'en' ? 'bg-primary text-white' : 'bg-gray-200 hover:bg-gray-300'
                        }`}
                    >
                        English
                    </button>
                </div>
            </div>

            <div className="space-y-6">
                <CategorySection title={t('expenseCategories')} tree={expenseTree} type="expense" />
                <CategorySection title={t('incomeCategories')} tree={incomeTree} type="income" />
                
                 <div className="bg-surface p-6 rounded-xl shadow-sm">
                    <h2 className="text-xl font-bold mb-4">{t('systemCategories')}</h2>
                    <div className="bg-gray-50 p-2 rounded-lg border divide-y">
                        {systemCategoriesList.map(cat => renderCategory(cat, 0))}
                    </div>
                </div>
            </div>


            <div className="bg-surface p-6 rounded-xl shadow-sm">
                <h2 className="text-xl font-bold mb-4">{t('dataManagement')}</h2>
                <div className="flex space-x-4">
                    <button 
                        onClick={handleBackup}
                        className="bg-secondary text-white px-4 py-2 rounded-lg hover:bg-green-600 transition-colors"
                    >
                        {t('backupData')}
                    </button>
                    <button 
                        onClick={handleRestoreClick}
                        className="border border-red-500 text-red-500 px-4 py-2 rounded-lg hover:bg-red-50 transition-colors"
                    >
                        {t('restoreData')}
                    </button>
                    <input 
                        type="file" 
                        ref={restoreInputRef}
                        className="hidden"
                        accept="application/json"
                        onChange={handleFileChange}
                    />
                </div>
            </div>
            
            <CategoryModal 
                isOpen={isModalOpen}
                onClose={() => setIsModalOpen(false)}
                initialData={editingCategory}
                categoryType={editingCategory ? getCategoryType(editingCategory.id) as CategoryType : newCategoryType}
            />
        </div>
    );
};

export default Settings;