import React, { useState } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { ChevronLeftIcon, ChevronRightIcon } from './icons/Icons';

const Calendar: React.FC = () => {
    const { transactions } = useData();
    const { language } = useLanguage();
    const [currentDate, setCurrentDate] = useState(new Date());

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat(language === 'vi' ? 'vi-VN' : 'en-US', {
            style: 'currency', currency: language === 'vi' ? 'VND' : 'USD'
        }).format(amount);
    };

    const daysOfWeek = language === 'vi' 
        ? ['CN', 'T2', 'T3', 'T4', 'T5', 'T6', 'T7'] 
        : ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];

    const firstDayOfMonth = new Date(currentDate.getFullYear(), currentDate.getMonth(), 1);
    const lastDayOfMonth = new Date(currentDate.getFullYear(), currentDate.getMonth() + 1, 0);
    const startingDay = firstDayOfMonth.getDay();
    const daysInMonth = lastDayOfMonth.getDate();

    const calendarDays = [];

    // Add empty cells for days before the 1st
    for (let i = 0; i < startingDay; i++) {
        calendarDays.push(<div key={`empty-${i}`} className="border-r border-b"></div>);
    }

    // Add day cells
    for (let day = 1; day <= daysInMonth; day++) {
        const dateStr = `${currentDate.getFullYear()}-${String(currentDate.getMonth() + 1).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
        
        const dailyTransactions = transactions.filter(t => t.date === dateStr);
        const dailyIncome = dailyTransactions.filter(t => t.type === 'income').reduce((sum, t) => sum + t.amount, 0);
        const dailyExpense = dailyTransactions.filter(t => t.type === 'expense').reduce((sum, t) => sum + t.amount, 0);
        
        const isToday = new Date().toDateString() === new Date(currentDate.getFullYear(), currentDate.getMonth(), day).toDateString();

        calendarDays.push(
            <div key={day} className="border-r border-b p-2 min-h-[100px] flex flex-col">
                <div className={`font-semibold text-sm ${isToday ? 'bg-primary text-white rounded-full w-6 h-6 flex items-center justify-center' : ''}`}>
                    {day}
                </div>
                <div className="mt-1 text-xs space-y-1 overflow-y-auto">
                    {dailyIncome > 0 && <p className="text-green-600 truncate font-semibold">+ {formatCurrency(dailyIncome)}</p>}
                    {dailyExpense > 0 && <p className="text-red-600 truncate font-semibold">- {formatCurrency(dailyExpense)}</p>}
                </div>
            </div>
        );
    }

    const nextMonth = () => setCurrentDate(new Date(currentDate.getFullYear(), currentDate.getMonth() + 1, 1));
    const prevMonth = () => setCurrentDate(new Date(currentDate.getFullYear(), currentDate.getMonth() - 1, 1));

    return (
        <div>
            <div className="flex justify-between items-center mb-4">
                <h3 className="text-lg font-bold">
                    {new Intl.DateTimeFormat(language === 'vi' ? 'vi-VN' : 'en-US', { month: 'long', year: 'numeric' }).format(currentDate)}
                </h3>
                <div className="flex gap-2">
                    <button onClick={prevMonth} className="p-1 rounded-full hover:bg-gray-100"><ChevronLeftIcon className="w-5 h-5" /></button>
                    <button onClick={nextMonth} className="p-1 rounded-full hover:bg-gray-100"><ChevronRightIcon className="w-5 h-5" /></button>
                </div>
            </div>
            <div className="grid grid-cols-7 border-t border-l">
                {daysOfWeek.map(day => (
                    <div key={day} className="text-center font-bold text-xs py-2 border-r border-b bg-gray-50 text-on-surface-secondary">{day}</div>
                ))}
                {calendarDays}
            </div>
        </div>
    );
};

export default Calendar;