
import React, { useState } from 'react';
import { DataProvider } from './context/DataContext';
import { LanguageProvider, useLanguage } from './context/LanguageContext';
import { ToastProvider } from './context/ToastContext';
import Dashboard from './components/Dashboard';
import Transactions from './components/Transactions';
import Settings from './components/Settings';
import Wallets from './components/Wallets';
import Toast from './components/Toast';
import DebtLedger from './components/DebtLedger';
import Reports from './components/Reports';
import { HomeIcon, ArrowTrendingUpIcon, Cog6ToothIcon, CreditCardIcon, SparklesIcon, BookOpenIcon, ChartBarIcon } from './components/icons/Icons';

type View = 'dashboard' | 'transactions' | 'reports' | 'settings' | 'wallets' | 'debtLedger';

const AppContent: React.FC = () => {
    const [activeView, setActiveView] = useState<View>('dashboard');
    const { t } = useLanguage();

    const renderView = () => {
        switch (activeView) {
            case 'dashboard': return <Dashboard />;
            case 'transactions': return <Transactions />;
            case 'reports': return <Reports />;
            case 'wallets': return <Wallets />;
            case 'debtLedger': return <DebtLedger />;
            case 'settings': return <Settings />;
            default: return <Dashboard />;
        }
    };
    
    const navItems = [
        { id: 'dashboard', icon: HomeIcon, label: t('dashboard') },
        { id: 'transactions', icon: ArrowTrendingUpIcon, label: t('transactions') },
        { id: 'reports', icon: ChartBarIcon, label: t('reports') },
        { id: 'wallets', icon: CreditCardIcon, label: t('wallets') },
        { id: 'debtLedger', icon: BookOpenIcon, label: t('debtLedger') },
        { id: 'settings', icon: Cog6ToothIcon, label: t('settings') },
    ];

    return (
        <div className="flex h-screen bg-background text-on-surface">
            <nav className="flex flex-col w-16 md:w-64 bg-surface border-r border-gray-200 transition-all duration-300 group">
                <div className="flex items-center justify-center md:justify-start md:px-6 h-20 border-b border-gray-200">
                   <div className="w-10 h-10 bg-primary rounded-lg flex items-center justify-center flex-shrink-0">
                        <SparklesIcon className="w-6 h-6 text-white" />
                   </div>
                   <h1 className="hidden md:block ml-3 text-xl font-bold text-primary whitespace-nowrap">{t('appName')}</h1>
                </div>
                <ul className="flex-grow pt-4">
                    {navItems.map(item => (
                        <li key={item.id} className="px-3 md:px-6 py-1">
                            <button
                                onClick={() => setActiveView(item.id as View)}
                                className={`flex items-center w-full h-12 rounded-lg transition-colors duration-200 ${
                                    activeView === item.id 
                                    ? 'bg-primary/10 text-primary font-semibold' 
                                    : 'text-on-surface-secondary hover:bg-gray-100'
                                }`}
                                title={item.label}
                            >
                                <item.icon className="h-6 w-6 mx-3 md:mx-4 flex-shrink-0" />
                                <span className="hidden md:block">{item.label}</span>
                            </button>
                        </li>
                    ))}
                </ul>
            </nav>

            <main className="flex-1 overflow-y-auto">
                <div className="p-4 sm:p-6 md:p-8">
                    {renderView()}
                </div>
            </main>
            <Toast />
        </div>
    );
}

const App: React.FC = () => {
    return (
        <LanguageProvider>
            <ToastProvider>
                <DataProvider>
                    <AppContent />
                </DataProvider>
            </ToastProvider>
        </LanguageProvider>
    );
};

export default App;
