
import React, { useMemo, useState, useEffect } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { useLocalStorage } from '../hooks/useLocalStorage';
import { ArrowDownTrayIcon, ChartBarIcon, DocumentTextIcon, HeartIcon, Icon } from './icons/Icons';
import { PieChart, Pie, Cell, ResponsiveContainer, Tooltip } from 'recharts';

const COLORS = [
    '#4f46e5', '#ec4899', '#10b981', '#f59e0b', '#8b5cf6', 
    '#06b6d4', '#ef4444', '#14b8a6', '#f43f5e', '#6366f1', 
    '#d946ef', '#facc15', '#78716c'
];

type TimeRange = 'thisMonth' | '30days' | '3months' | 'custom';

const Reports: React.FC = () => {
    const { netTransactionsForReporting, categories, wallets, totalBalance } = useData();
    const { t, language } = useLanguage();

    // Persistent comparison categories
    const [selectedCategoryIds, setSelectedCategoryIds] = useLocalStorage<string[]>('comparison_categories', []);
    
    const [timeRange, setTimeRange] = useState<TimeRange>('thisMonth');
    const [customStartDate, setCustomStartDate] = useState('');
    const [customEndDate, setCustomEndDate] = useState('');

    // Initialize defaults only if local storage is truly empty
    useEffect(() => {
        if (selectedCategoryIds.length === 0 && categories.length > 0) {
            const topLevelExpenses = categories.filter(c => !c.parentId && c.id !== 'cat-i1' && c.id !== 'cat-transfer' && c.id !== 'cat-debt');
            setSelectedCategoryIds(topLevelExpenses.map(c => c.id));
        }
    }, [categories]);

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat(language === 'vi' ? 'vi-VN' : 'en-US', {
            style: 'currency', currency: language === 'vi' ? 'VND' : 'USD'
        }).format(amount);
    };

    const financialHealthData = useMemo(() => {
        const now = new Date();
        const thisMonthKey = `${now.getFullYear()}-${String(now.getMonth() + 1).padStart(2, '0')}`;
        
        let monthIncome = 0;
        let monthExpense = 0;
        let needs = 0;
        let wants = 0;
        
        const needsCategories = ['cat-e11', 'cat-e13', 'cat-e13a', 'cat-e13b', 'cat-e2', 'cat-e5', 'cat-e15'];
        const wantsCategories = ['cat-e3', 'cat-e4', 'cat-e8', 'cat-e14'];

        netTransactionsForReporting.forEach(tx => {
            const tDate = new Date(`${tx.date}T00:00:00`);
            const mKey = `${tDate.getFullYear()}-${String(tDate.getMonth() + 1).padStart(2, '0')}`;
            
            if (mKey === thisMonthKey) {
                if (tx.type === 'income') {
                    monthIncome += tx.amount;
                } else {
                    monthExpense += tx.amount;
                    let currentCat = categories.find(c => c.id === tx.categoryId);
                    const isNeed = needsCategories.includes(tx.categoryId) || (currentCat?.parentId && needsCategories.includes(currentCat.parentId));
                    const isWant = wantsCategories.includes(tx.categoryId) || (currentCat?.parentId && wantsCategories.includes(currentCat.parentId));
                    if (isNeed) needs += tx.amount;
                    else if (isWant) wants += tx.amount;
                }
            }
        });

        const savings = Math.max(0, monthIncome - monthExpense);
        const savingsRate = monthIncome > 0 ? (savings / monthIncome) * 100 : 0;
        
        let message = '';
        if (monthIncome === 0 && monthExpense > 0) {
            message = language === 'vi' ? 'Bạn chưa ghi nhận thu nhập tháng này.' : 'No income recorded this month.';
        } else if (savingsRate > 30) {
            message = language === 'vi' ? 'Tuyệt vời! Tỷ lệ tiết kiệm của bạn rất cao.' : 'Excellent! Your savings rate is very high.';
        } else if (savingsRate >= 15) {
            message = language === 'vi' ? 'Tốt. Bạn đang duy trì mức tích lũy an toàn.' : 'Good. You are maintaining a safe surplus.';
        } else {
            message = language === 'vi' ? 'Hãy chú ý hơn đến các khoản chi không thiết yếu.' : 'Watch your non-essential spending.';
        }

        const ruleData = [
            { name: language === 'vi' ? 'Thiết yếu' : 'Needs', value: needs, color: '#4f46e5' },
            { name: language === 'vi' ? 'Sở thích' : 'Wants', value: wants, color: '#ec4899' },
            { name: language === 'vi' ? 'Tích lũy' : 'Savings', value: savings, color: '#10b981' }
        ].filter(d => d.value > 0);

        return { monthIncome, monthExpense, savingsRate, message, ruleData, savings };
    }, [netTransactionsForReporting, categories, language]);

    const comparisonData = useMemo(() => {
        if (selectedCategoryIds.length === 0) return [];

        const now = new Date();
        now.setHours(23, 59, 59, 999);
        let startDate: Date | null = null;

        switch(timeRange) {
            case 'thisMonth':
                startDate = new Date(now.getFullYear(), now.getMonth(), 1);
                break;
            case '30days':
                startDate = new Date();
                startDate.setDate(now.getDate() - 30);
                break;
            case '3months':
                startDate = new Date();
                startDate.setMonth(now.getMonth() - 3);
                break;
            case 'custom':
                if (customStartDate) startDate = new Date(`${customStartDate}T00:00:00`);
                break;
        }

        const endDate = (timeRange === 'custom' && customEndDate) 
            ? new Date(`${customEndDate}T23:59:59`) 
            : now;

        const data: { [key: string]: number } = {};
        const isDescendantOfSelected = (catId: string): string | null => {
            let current = categories.find(c => c.id === catId);
            while (current) {
                if (selectedCategoryIds.includes(current.id)) return current.id;
                current = categories.find(c => c.id === current?.parentId);
            }
            return null;
        };

        netTransactionsForReporting
            .filter(t => t.type === 'expense' && t.categoryId !== 'cat-transfer')
            .filter(t => {
                const d = new Date(`${t.date}T12:00:00`);
                if (startDate && d < startDate) return false;
                if (endDate && d > endDate) return false;
                return true;
            })
            .forEach(t => {
                const parentSelectedId = isDescendantOfSelected(t.categoryId);
                if (parentSelectedId) {
                    const catName = categories.find(c => c.id === parentSelectedId)?.name || 'Other';
                    data[catName] = (data[catName] || 0) + t.amount;
                }
            });

        return Object.entries(data)
            .map(([name, value]) => ({ name, value }))
            .sort((a, b) => b.value - a.value);
    }, [netTransactionsForReporting, categories, selectedCategoryIds, timeRange, customStartDate, customEndDate]);

    const handleToggleCategory = (id: string) => {
        if (selectedCategoryIds.includes(id)) {
            setSelectedCategoryIds(selectedCategoryIds.filter(cid => cid !== id));
        } else {
            setSelectedCategoryIds([...selectedCategoryIds, id]);
        }
    };

    const handleReset = () => {
        const topLevelExpenses = categories.filter(c => !c.parentId && c.id !== 'cat-i1' && c.id !== 'cat-transfer' && c.id !== 'cat-debt');
        setSelectedCategoryIds(topLevelExpenses.map(c => c.id));
    };

    const exportDetailedCSV = () => {
        const headers = language === 'vi' ? 'Ngày,Mô tả,Số tiền,Loại,Danh mục,Ví' : 'Date,Description,Amount,Type,Category,Wallet';
        const rows = netTransactionsForReporting.map(tx => {
            const cat = categories.find(c => c.id === tx.categoryId)?.name || '';
            const wallet = wallets.find(w => w.id === tx.walletId)?.name || '';
            return `${tx.date},"${tx.description}",${tx.amount},${tx.type},"${cat}","${wallet}"`;
        });
        const blob = new Blob([`\ufeff${headers}\n${rows.join('\n')}`], { type: 'text/csv;charset=utf-8;' });
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.setAttribute('download', `wolf_ledger_${new Date().toISOString().split('T')[0]}.csv`);
        link.click();
    };

    return (
        <div className="space-y-8">
            <h1 className="text-3xl font-bold text-on-surface">{t('reports')}</h1>

            {/* Health Card */}
            <div className="bg-surface p-6 rounded-2xl shadow-sm border border-gray-100">
                <div className="flex items-center gap-4 mb-6">
                    <div className="p-3 bg-red-100 rounded-xl">
                        <HeartIcon className="w-8 h-8 text-red-500" />
                    </div>
                    <div>
                        <h2 className="text-xl font-bold">{language === 'vi' ? 'Sức khỏe Tài chính' : 'Financial Health'}</h2>
                        <p className="text-on-surface-secondary text-sm">{financialHealthData.message}</p>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    <div className="space-y-4">
                        <div className="p-4 bg-gray-50 rounded-xl">
                            <p className="text-xs text-on-surface-secondary uppercase font-bold">{t('income')}</p>
                            <p className="text-xl font-bold text-green-600">{formatCurrency(financialHealthData.monthIncome)}</p>
                        </div>
                        <div className="p-4 bg-gray-50 rounded-xl">
                            <p className="text-xs text-on-surface-secondary uppercase font-bold">{t('expense')}</p>
                            <p className="text-xl font-bold text-red-500">{formatCurrency(financialHealthData.monthExpense)}</p>
                        </div>
                        <div className="p-4 bg-gray-50 rounded-xl">
                            <p className="text-xs text-on-surface-secondary uppercase font-bold">{language === 'vi' ? 'Tỷ lệ Tiết kiệm' : 'Savings Rate'}</p>
                            <p className="text-xl font-bold text-primary">{financialHealthData.savingsRate.toFixed(1)}%</p>
                        </div>
                    </div>
                    <div className="lg:col-span-2 flex flex-col md:flex-row items-center gap-6 bg-gray-50/30 p-6 rounded-xl border border-gray-100">
                        <div className="w-full h-[220px] md:w-1/2">
                            <ResponsiveContainer width="100%" height="100%">
                                <PieChart>
                                    <Pie data={financialHealthData.ruleData} cx="50%" cy="50%" innerRadius={65} outerRadius={85} paddingAngle={5} dataKey="value">
                                        {financialHealthData.ruleData.map((entry, index) => <Cell key={`cell-${index}`} fill={entry.color} />)}
                                    </Pie>
                                    <Tooltip formatter={(val) => formatCurrency(val as number)} />
                                </PieChart>
                            </ResponsiveContainer>
                        </div>
                        <div className="w-full md:w-1/2 space-y-3">
                            <h3 className="font-bold text-sm">{language === 'vi' ? 'Phân bổ 50/30/20' : '50/30/20 Breakdown'}</h3>
                            {financialHealthData.ruleData.map((d, i) => (
                                <div key={i} className="flex justify-between items-center text-sm">
                                    <div className="flex items-center gap-2">
                                        <div className="w-3 h-3 rounded-full" style={{ backgroundColor: d.color }}></div>
                                        <span>{d.name}</span>
                                    </div>
                                    <span className="font-bold">{financialHealthData.monthExpense > 0 ? ((d.value / (financialHealthData.monthIncome || financialHealthData.monthExpense)) * 100).toFixed(0) : 0}%</span>
                                </div>
                            ))}
                        </div>
                    </div>
                </div>
            </div>

            {/* Custom Comparison Section */}
            <div className="bg-surface p-6 sm:p-8 rounded-2xl shadow-sm border border-gray-100 h-auto overflow-visible">
                <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-6 mb-8">
                    <div className="flex items-center gap-3">
                        <div className="p-2 bg-primary/10 rounded-lg">
                            <ChartBarIcon className="w-6 h-6 text-primary" />
                        </div>
                        <h2 className="text-2xl font-bold text-on-surface">{t('categoryComparison')}</h2>
                    </div>
                    
                    <div className="flex flex-wrap items-center gap-3">
                        <div className="flex bg-gray-100 p-1 rounded-xl shadow-inner">
                            {(['thisMonth', '30days', '3months', 'custom'] as TimeRange[]).map((range) => (
                                <button 
                                    key={range}
                                    onClick={() => setTimeRange(range)} 
                                    className={`px-3 py-1.5 text-xs font-bold rounded-lg transition-all ${timeRange === range ? 'bg-white shadow text-primary' : 'text-on-surface-secondary hover:text-on-surface'}`}
                                >
                                    {t(range === 'thisMonth' ? 'thisMonth' : range === '30days' ? 'last30Days' : range === '3months' ? 'last3Months' : 'customRange')}
                                </button>
                            ))}
                        </div>
                        
                        {timeRange === 'custom' && (
                            <div className="flex items-center gap-2 animate-in slide-in-from-right duration-300">
                                <input type="date" value={customStartDate} onChange={(e) => setCustomStartDate(e.target.value)} className="px-2 py-1.5 text-xs border rounded-md focus:ring-1 focus:ring-primary outline-none" />
                                <span className="text-xs text-gray-400">→</span>
                                <input type="date" value={customEndDate} onChange={(e) => setCustomEndDate(e.target.value)} className="px-2 py-1.5 text-xs border rounded-md focus:ring-1 focus:ring-primary outline-none" />
                            </div>
                        )}
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-12 gap-8 items-start h-auto overflow-visible">
                    {/* Left side: Categories - Fully visible, no overflow */}
                    <div className="lg:col-span-4 space-y-6 h-auto">
                        <div className="flex justify-between items-center">
                            <h3 className="text-xs font-bold text-on-surface-secondary uppercase tracking-widest">{t('selectCategories')}</h3>
                            <button onClick={handleReset} className="text-[10px] font-bold text-primary hover:bg-primary/5 px-2 py-1 rounded transition-colors">Reset</button>
                        </div>
                        <div className="flex flex-wrap gap-2 h-auto overflow-visible">
                            {categories
                                .filter(c => !c.parentId && c.id !== 'cat-i1' && c.id !== 'cat-transfer' && c.id !== 'cat-debt')
                                .map(cat => (
                                    <button
                                        key={cat.id}
                                        onClick={() => handleToggleCategory(cat.id)}
                                        className={`px-4 py-2.5 rounded-2xl text-xs font-bold transition-all border-2 flex items-center gap-2 ${
                                            selectedCategoryIds.includes(cat.id) 
                                            ? 'bg-primary border-primary text-white shadow-lg shadow-primary/20 scale-[1.02]' 
                                            : 'bg-white border-gray-100 text-on-surface-secondary hover:border-primary/30 hover:text-primary'
                                        }`}
                                    >
                                        <Icon name={cat.icon} className="w-4 h-4" />
                                        {cat.name}
                                    </button>
                                ))}
                        </div>
                    </div>

                    {/* Right side: Chart - Expands with content */}
                    <div className="lg:col-span-8 bg-gray-50/50 rounded-3xl p-6 sm:p-10 border border-gray-100 min-h-[450px] flex flex-col h-auto overflow-visible">
                        {comparisonData.length > 0 ? (
                            <div className="flex-1 flex flex-col md:flex-row items-center gap-10 h-auto">
                                <div className="w-full h-[320px] md:w-1/2 flex-shrink-0">
                                    <ResponsiveContainer width="100%" height="100%">
                                        <PieChart>
                                            <Pie data={comparisonData} cx="50%" cy="50%" innerRadius={80} outerRadius={115} paddingAngle={4} dataKey="value" animationDuration={1000}>
                                                {comparisonData.map((_, index) => <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} strokeWidth={0} />)}
                                            </Pie>
                                            <Tooltip formatter={(val) => formatCurrency(val as number)} />
                                        </PieChart>
                                    </ResponsiveContainer>
                                </div>
                                <div className="w-full md:w-1/2 h-auto">
                                    <div className="bg-white rounded-2xl p-6 shadow-sm border border-gray-100 h-auto overflow-visible">
                                        <p className="text-xs text-on-surface-secondary uppercase font-bold mb-5 tracking-widest">{t('allExpenses')}</p>
                                        <div className="space-y-4 h-auto">
                                            {comparisonData.map((d, i) => (
                                                <div key={i} className="flex justify-between items-start text-sm">
                                                    <div className="flex items-start gap-3 mt-1 min-w-0">
                                                        <div className="w-2.5 h-2.5 rounded-full mt-1.5 flex-shrink-0" style={{ backgroundColor: COLORS[i % COLORS.length] }}></div>
                                                        <span className="font-semibold text-on-surface leading-tight break-words">{d.name}</span>
                                                    </div>
                                                    <div className="text-right flex-shrink-0 ml-4">
                                                        <p className="font-bold text-on-surface">{formatCurrency(d.value)}</p>
                                                        <p className="text-[10px] text-on-surface-secondary font-bold">
                                                            {((d.value / comparisonData.reduce((s, x) => s + x.value, 0)) * 100).toFixed(1)}%
                                                        </p>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                </div>
                            </div>
                        ) : (
                            <div className="flex-1 flex flex-col items-center justify-center text-on-surface-secondary text-center space-y-6">
                                <div className="p-8 bg-white rounded-full shadow-inner border border-gray-100">
                                    <ChartBarIcon className="w-16 h-16 text-gray-200" />
                                </div>
                                <div>
                                    <p className="font-bold text-xl text-on-surface">{language === 'vi' ? 'Chưa có dữ liệu' : 'No data'}</p>
                                    <p className="text-sm max-w-xs mt-2">{language === 'vi' ? 'Hãy chọn danh mục và khoảng thời gian có phát sinh chi tiêu.' : 'Select categories and a time range with expenses.'}</p>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>

            {/* Export and Info Cards */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div className="bg-surface p-6 rounded-2xl shadow-sm border border-gray-100 flex flex-col justify-between">
                    <div className="flex items-start gap-4">
                        <div className="p-3 bg-primary/10 rounded-lg">
                            <DocumentTextIcon className="w-8 h-8 text-primary" />
                        </div>
                        <div>
                            <h2 className="text-xl font-bold">{t('detailedReport')}</h2>
                            <p className="text-on-surface-secondary text-sm mt-1">Tải về toàn bộ lịch sử giao dịch để phân tích.</p>
                        </div>
                    </div>
                    <button onClick={exportDetailedCSV} className="mt-6 flex items-center justify-center gap-2 bg-primary text-white py-3 rounded-xl hover:bg-primary-dark transition-all font-bold shadow-lg shadow-primary/20">
                        <ArrowDownTrayIcon className="w-5 h-5" />
                        <span>{t('exportCSV')}</span>
                    </button>
                </div>

                <div className="bg-surface p-6 rounded-2xl shadow-sm border border-gray-100">
                    <div className="flex items-start gap-4 mb-6">
                        <div className="p-3 bg-secondary/10 rounded-lg">
                            <ChartBarIcon className="w-8 h-8 text-secondary" />
                        </div>
                        <div>
                            <h2 className="text-xl font-bold">{language === 'vi' ? 'Quỹ Dự phòng' : 'Emergency Fund'}</h2>
                            <p className="text-on-surface-secondary text-sm mt-1">
                                {language === 'vi' 
                                    ? `Bạn có thể duy trì cuộc sống trong ${(totalBalance / (financialHealthData.monthExpense || 1)).toFixed(1)} tháng.` 
                                    : `You can sustain for ${(totalBalance / (financialHealthData.monthExpense || 1)).toFixed(1)} months.`}
                            </p>
                        </div>
                    </div>
                    <div className="w-full bg-gray-100 h-2.5 rounded-full overflow-hidden">
                        <div className="bg-secondary h-full transition-all duration-1000" style={{ width: `${Math.min((totalBalance / (financialHealthData.monthExpense * 6 || 1)) * 100, 100)}%` }}></div>
                    </div>
                    <p className="text-[10px] text-on-surface-secondary mt-2 text-right font-bold uppercase tracking-widest">{language === 'vi' ? 'Mục tiêu: 6 tháng chi tiêu' : 'Goal: 6 months expenses'}</p>
                </div>
            </div>
        </div>
    );
};

export default Reports;
