import React, { useState, useEffect, FormEvent, useMemo } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Debt } from '../types';
import { XMarkIcon } from './icons/Icons';

interface RepaymentModalProps {
    isOpen: boolean;
    onClose: () => void;
    debt: Debt;
}

const RepaymentModal: React.FC<RepaymentModalProps> = ({ isOpen, onClose, debt }) => {
    const { wallets, addRepayment } = useData();
    const { t, formatDate, parseDate, formatCurrencyInput, parseCurrencyInput } = useLanguage();
    
    const [amount, setAmount] = useState('');
    const [date, setDate] = useState(''); // YYYY-MM-DD
    const [displayDate, setDisplayDate] = useState(''); // DD/MM/YYYY
    const [walletId, setWalletId] = useState('');
    const [error, setError] = useState('');

    const sortedWallets = useMemo(() => {
        return [...wallets].sort((a, b) => a.order - b.order);
    }, [wallets]);

    useEffect(() => {
        if (isOpen) {
            const today = new Date().toISOString().split('T')[0];
            setAmount('');
            setDate(today);
            setDisplayDate(formatDate(today));
            setWalletId(sortedWallets.length > 0 ? sortedWallets[0].id : '');
            setError('');
        }
    }, [isOpen, sortedWallets, formatDate]);

    const handleDateChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const newDisplayDate = e.target.value;
        setDisplayDate(newDisplayDate);
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            setDate(newIsoDate);
        }
    };

    const handleDateBlur = (e: React.FocusEvent<HTMLInputElement>) => {
        const newDisplayDate = e.target.value;
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            setDate(newIsoDate);
            setDisplayDate(formatDate(newIsoDate));
        } else if (newDisplayDate === '') {
            setDate('');
        } else {
            setDisplayDate(formatDate(date));
        }
    };

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        const numAmount = parseCurrencyInput(amount);
        if (!amount || !date || !walletId) {
            setError(t('allFieldsRequired'));
            return;
        }
        if (isNaN(numAmount) || numAmount <= 0) {
            setError(t('validAmount'));
            return;
        }
        if (numAmount > debt.outstandingAmount) {
            setError(`Amount cannot be greater than the outstanding amount of ${debt.outstandingAmount}.`);
            return;
        }
        
        addRepayment({
            debtId: debt.id,
            amount: numAmount,
            date,
            walletId,
        });
        onClose();
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center z-50">
            <div className="bg-surface rounded-xl shadow-2xl w-full max-w-md m-4">
                <form onSubmit={handleSubmit}>
                    <div className="p-6 border-b border-gray-200 flex justify-between items-center">
                        <h2 className="text-xl font-bold">{t('recordRepayment')}</h2>
                        <button type="button" onClick={onClose} className="text-on-surface-secondary hover:text-on-surface">
                            <XMarkIcon className="w-6 h-6" />
                        </button>
                    </div>

                    <div className="p-6 space-y-4">
                        <p className='text-sm text-on-surface-secondary'>{t('repaymentTo')} <span className='font-bold text-on-surface'>{debt.person}</span></p>
                        <div>
                            <label htmlFor="repaymentAmount" className="block text-sm font-medium text-on-surface-secondary">{t('repaymentAmount')}</label>
                            <input 
                                type="text" 
                                inputMode="numeric" 
                                id="repaymentAmount" 
                                value={amount} 
                                onChange={e => setAmount(formatCurrencyInput(e.target.value))} 
                                placeholder="0" 
                                className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                            />
                        </div>
                        <div className="grid grid-cols-2 gap-4">
                            <div>
                                <label htmlFor="repaymentDate" className="block text-sm font-medium text-on-surface-secondary">{t('repaymentDate')}</label>
                                <input 
                                    type="text" 
                                    id="repaymentDate" 
                                    value={displayDate} 
                                    onChange={handleDateChange} 
                                    onBlur={handleDateBlur}
                                    placeholder="dd/mm/yyyy"
                                    className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                                />
                            </div>
                            <div>
                                <label htmlFor="wallet" className="block text-sm font-medium text-on-surface-secondary">{t('wallet')}</label>
                                <select id="wallet" value={walletId} onChange={e => setWalletId(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                    <option value="" disabled>{t('selectWallet')}</option>
                                    {sortedWallets.map(w => (
                                        <option key={w.id} value={w.id}>{w.name}</option>
                                    ))}
                                </select>
                            </div>
                        </div>
                        {error && <p className="text-sm text-red-600">{error}</p>}
                    </div>

                    <div className="p-6 bg-gray-50 rounded-b-xl flex justify-end items-center gap-4">
                        <button type="button" onClick={onClose} className="px-4 py-2 text-sm font-semibold text-on-surface-secondary bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                            {t('cancel')}
                        </button>
                        <button type="submit" className="px-4 py-2 text-sm font-semibold text-white bg-primary rounded-lg hover:bg-primary-dark transition-colors">
                            {t('save')}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default RepaymentModal;