
import React, { useState, useEffect, FormEvent, useMemo } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Category } from '../types';
import { XMarkIcon, Icon, iconLibrary } from './icons/Icons';

type CategoryType = 'income' | 'expense';

interface CategoryModalProps {
    isOpen: boolean;
    onClose: () => void;
    initialData?: Category | null;
    categoryType: CategoryType;
}

const iconNames = Object.keys(iconLibrary).filter(name => !['HomeIcon', 'ArrowTrendingUpIcon', 'Cog6ToothIcon', 'CreditCardIcon', 'BookOpenIcon', 'ChevronLeftIcon', 'ChevronRightIcon', 'XMarkIcon', 'PlusIcon', 'PencilSquareIcon', 'TrashIcon', 'Bars3Icon'].includes(name));
const colors = ['#ef4444', '#f97316', '#eab308', '#84cc16', '#22c55e', '#10b981', '#14b8a6', '#06b6d4', '#0ea5e9', '#3b82f6', '#6366f1', '#8b5cf6', '#a855f7', '#d946ef', '#ec4899', '#f43f5e', '#78716c'];

const CategoryModal: React.FC<CategoryModalProps> = ({ isOpen, onClose, initialData, categoryType }) => {
    const { categories, addCategory, updateCategory } = useData();
    const { t } = useLanguage();
    
    const [name, setName] = useState('');
    const [parentId, setParentId] = useState<string | null>(null);
    const [icon, setIcon] = useState('SparklesIcon');
    const [color, setColor] = useState('#6366f1');
    const [error, setError] = useState('');

    useEffect(() => {
        if (isOpen) {
            if (initialData) {
                setName(initialData.name);
                setParentId(initialData.parentId || null);
                setIcon(initialData.icon);
                setColor(initialData.color);
            } else {
                setName('');
                // If adding a new income sub-category, default its parent to the main income category
                setParentId(categoryType === 'income' ? 'cat-i1' : null);
                setIcon('SparklesIcon');
                setColor('#6366f1');
            }
            setError('');
        }
    }, [initialData, isOpen, categoryType]);

    const parentCategoryOptions = useMemo(() => {
        const systemIds = ['cat-transfer', 'cat-debt'];
        const incomeRootId = 'cat-i1';

        const categoryMap = new Map<string, Category[]>();
        const rootCategories: Category[] = [];

        categories.forEach(c => {
            if (systemIds.includes(c.id)) return;
            
            if (c.parentId) {
                if (!categoryMap.has(c.parentId)) {
                    categoryMap.set(c.parentId, []);
                }
                categoryMap.get(c.parentId)!.push(c);
            } else {
                rootCategories.push(c);
            }
        });

        const getCategoryType = (cat: Category): CategoryType => {
            let current = cat;
            while(current.parentId) {
                const parent = categories.find(c => c.id === current.parentId);
                if (!parent) return 'expense'; // Should not happen
                current = parent;
            }
            return current.id === incomeRootId ? 'income' : 'expense';
        }

        return categories
            .filter(c => {
                if(c.id === initialData?.id || systemIds.includes(c.id)) return false; // Can't be its own parent
                return getCategoryType(c) === categoryType;
            })
            .filter(c => c.id !== (categoryType === 'expense' ? incomeRootId : undefined))


    }, [categories, initialData, categoryType]);

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        if (!name) {
            setError(t('allFieldsRequired'));
            return;
        }
        
        const categoryData = { name, parentId, icon, color };
        
        if (initialData) {
            updateCategory({ ...categoryData, id: initialData.id, order: initialData.order });
        } else {
            addCategory(categoryData);
        }
        onClose();
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center z-50">
            <div className="bg-surface rounded-xl shadow-2xl w-full max-w-md m-4">
                <form onSubmit={handleSubmit}>
                    <div className="p-6 border-b border-gray-200 flex justify-between items-center">
                        <h2 className="text-xl font-bold">{initialData ? t('editCategory') : t('addCategory')}</h2>
                        <button type="button" onClick={onClose} className="text-on-surface-secondary hover:text-on-surface">
                            <XMarkIcon className="w-6 h-6" />
                        </button>
                    </div>

                    <div className="p-6 space-y-4 max-h-[70vh] overflow-y-auto">
                        <div>
                            <label htmlFor="name" className="block text-sm font-medium text-on-surface-secondary">{t('categoryName')}</label>
                            <input type="text" id="name" value={name} onChange={e => setName(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                        </div>
                        <div>
                             <label htmlFor="parentCategory" className="block text-sm font-medium text-on-surface-secondary">{t('parentCategory')}</label>
                            <select 
                                id="parentCategory" 
                                value={parentId || ''} 
                                onChange={e => setParentId(e.target.value || null)} 
                                className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface"
                                disabled={initialData?.id === 'cat-i1'}
                            >
                                <option value="">{t('noParent')}</option>
                                {parentCategoryOptions.map(cat => (
                                    <option key={cat.id} value={cat.id}>{cat.name}</option>
                                ))}
                            </select>
                        </div>
                         <div>
                            <label className="block text-sm font-medium text-on-surface-secondary">{t('icon')}</label>
                            <div className="mt-2 grid grid-cols-8 gap-2 border p-2 rounded-lg max-h-32 overflow-y-auto">
                                {iconNames.map(iconName => (
                                    <button 
                                        type="button" 
                                        key={iconName} 
                                        onClick={() => setIcon(iconName)}
                                        className={`flex items-center justify-center p-2 rounded-lg ${icon === iconName ? 'bg-primary/20 ring-2 ring-primary' : 'hover:bg-gray-100'}`}
                                    >
                                        <Icon name={iconName} className="w-5 h-5 text-on-surface" />
                                    </button>
                                ))}
                            </div>
                        </div>
                         <div>
                            <label className="block text-sm font-medium text-on-surface-secondary">{t('color')}</label>
                             <div className="mt-2 flex flex-wrap gap-2">
                                {colors.map(c => (
                                    <button
                                        type="button"
                                        key={c}
                                        onClick={() => setColor(c)}
                                        className={`w-8 h-8 rounded-full ${color === c ? 'ring-2 ring-primary ring-offset-2' : ''}`}
                                        style={{ backgroundColor: c }}
                                    />
                                ))}
                            </div>
                        </div>
                        {error && <p className="text-sm text-red-600">{error}</p>}
                    </div>

                    <div className="p-6 bg-gray-50 rounded-b-xl flex justify-end items-center gap-4">
                        <button type="button" onClick={onClose} className="px-4 py-2 text-sm font-semibold text-on-surface-secondary bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                            {t('cancel')}
                        </button>
                        <button type="submit" className="px-4 py-2 text-sm font-semibold text-white bg-primary rounded-lg hover:bg-primary-dark transition-colors">
                            {t('save')}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default CategoryModal;
