
import React, { useState, useEffect, FormEvent } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Debt } from '../types';
import { XMarkIcon } from './icons/Icons';

interface DebtModalProps {
    isOpen: boolean;
    onClose: () => void;
    defaultType: 'loan' | 'debt';
}

const DebtModal: React.FC<DebtModalProps> = ({ isOpen, onClose, defaultType }) => {
    const { addDebt } = useData();
    const { t, formatDate, parseDate, formatCurrencyInput, parseCurrencyInput } = useLanguage();
    
    const [type, setType] = useState<Debt['type']>(defaultType);
    const [person, setPerson] = useState('');
    const [initialAmount, setInitialAmount] = useState('');
    const [creationDate, setCreationDate] = useState(''); // YYYY-MM-DD
    const [displayCreationDate, setDisplayCreationDate] = useState(''); // DD/MM/YYYY
    const [dueDate, setDueDate] = useState(''); // YYYY-MM-DD
    const [displayDueDate, setDisplayDueDate] = useState(''); // DD/MM/YYYY
    const [description, setDescription] = useState('');
    const [error, setError] = useState('');

    useEffect(() => {
        if (isOpen) {
            const today = new Date().toISOString().split('T')[0];
            setType(defaultType);
            setPerson('');
            setInitialAmount('');
            setCreationDate(today);
            setDisplayCreationDate(formatDate(today));
            setDueDate('');
            setDisplayDueDate('');
            setDescription('');
            setError('');
        }
    }, [isOpen, defaultType, formatDate]);

    const handleDateChange = (e: React.ChangeEvent<HTMLInputElement>, dateSetter: React.Dispatch<React.SetStateAction<string>>, displaySetter: React.Dispatch<React.SetStateAction<string>>) => {
        const newDisplayDate = e.target.value;
        displaySetter(newDisplayDate);
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            dateSetter(newIsoDate);
        }
    };

    const handleDateBlur = (e: React.FocusEvent<HTMLInputElement>, dateValue: string, dateSetter: React.Dispatch<React.SetStateAction<string>>, displaySetter: React.Dispatch<React.SetStateAction<string>>) => {
        const newDisplayDate = e.target.value;
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            dateSetter(newIsoDate);
            displaySetter(formatDate(newIsoDate));
        } else if (newDisplayDate === '') {
            dateSetter('');
        } else {
            displaySetter(formatDate(dateValue));
        }
    };

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        const numAmount = parseCurrencyInput(initialAmount);
        if (!person || !initialAmount || !creationDate) {
            setError(t('allFieldsRequired'));
            return;
        }
        if (isNaN(numAmount) || numAmount <= 0) {
            setError(t('validAmount'));
            return;
        }
        
        addDebt({
            type,
            person,
            initialAmount: numAmount,
            creationDate,
            dueDate: dueDate || undefined,
            description
        });
        onClose();
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center z-50">
            <div className="bg-surface rounded-xl shadow-2xl w-full max-w-md m-4">
                <form onSubmit={handleSubmit}>
                    <div className="p-6 border-b border-gray-200 flex justify-between items-center">
                        <h2 className="text-xl font-bold">{type === 'loan' ? t('addLoan') : t('addDebt')}</h2>
                        <button type="button" onClick={onClose} className="text-on-surface-secondary hover:text-on-surface">
                            <XMarkIcon className="w-6 h-6" />
                        </button>
                    </div>

                    <div className="p-6 space-y-4 max-h-[70vh] overflow-y-auto">
                        <div>
                            <label htmlFor="person" className="block text-sm font-medium text-on-surface-secondary">{t('person')}</label>
                            <input type="text" id="person" value={person} onChange={e => setPerson(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                        </div>
                         <div>
                            <label htmlFor="amount" className="block text-sm font-medium text-on-surface-secondary">{t('amount')}</label>
                            <input 
                                type="text" 
                                inputMode="numeric" 
                                id="amount" 
                                value={initialAmount} 
                                onChange={e => setInitialAmount(formatCurrencyInput(e.target.value))} 
                                placeholder="0" 
                                className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                            />
                        </div>
                         <div className="grid grid-cols-2 gap-4">
                            <div>
                                <label htmlFor="creationDate" className="block text-sm font-medium text-on-surface-secondary">{t('creationDate')}</label>
                                <input 
                                    type="text" 
                                    id="creationDate" 
                                    value={displayCreationDate} 
                                    onChange={(e) => handleDateChange(e, setCreationDate, setDisplayCreationDate)}
                                    onBlur={(e) => handleDateBlur(e, creationDate, setCreationDate, setDisplayCreationDate)}
                                    placeholder="dd/mm/yyyy"
                                    className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                                />
                            </div>
                            <div>
                                {/* FIX: Replaced hardcoded string with translation key. */}
                                <label htmlFor="dueDate" className="block text-sm font-medium text-on-surface-secondary">{t('dueDate')} <span className='text-xs'>({t('optional')})</span></label>
                                <input 
                                    type="text" 
                                    id="dueDate" 
                                    value={displayDueDate} 
                                    onChange={(e) => handleDateChange(e, setDueDate, setDisplayDueDate)}
                                    onBlur={(e) => handleDateBlur(e, dueDate, setDueDate, setDisplayDueDate)}
                                    placeholder="dd/mm/yyyy"
                                    className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" 
                                />
                            </div>
                        </div>
                        <div>
                            {/* FIX: Replaced hardcoded string with translation key. */}
                            <label htmlFor="description" className="block text-sm font-medium text-on-surface-secondary">{t('description')} <span className='text-xs'>({t('optional')})</span></label>
                            <input type="text" id="description" value={description} onChange={e => setDescription(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                        </div>
                        {error && <p className="text-sm text-red-600">{error}</p>}
                    </div>

                    <div className="p-6 bg-gray-50 rounded-b-xl flex justify-end items-center gap-4">
                        <button type="button" onClick={onClose} className="px-4 py-2 text-sm font-semibold text-on-surface-secondary bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                            {t('cancel')}
                        </button>
                        <button type="submit" className="px-4 py-2 text-sm font-semibold text-white bg-primary rounded-lg hover:bg-primary-dark transition-colors">
                            {t('save')}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default DebtModal;
