
import { GoogleGenAI } from "@google/genai";
import { Transaction, Category } from '../types';

// FIX: Initialized GoogleGenAI directly using process.env.API_KEY as per instructions.
const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });

const formatTransactionsForPrompt = (transactions: Transaction[], categories: Category[]): string => {
    return transactions.slice(0, 50).map(t => {
        const categoryName = categories.find(c => c.id === t.categoryId)?.name || 'Unknown';
        return `${t.date}: ${t.type === 'expense' ? '-' : '+'}${t.amount} for ${t.description} (Category: ${categoryName})`;
    }).join('\n');
};

export const getSavingsSuggestion = async (transactions: Transaction[], categories: Category[], language: string): Promise<string> => {
    if (transactions.filter(t => t.type === 'expense').length < 5) {
        return language === 'vi' 
            ? 'Không đủ dữ liệu chi tiêu để đưa ra gợi ý. Hãy thêm ít nhất 5 giao dịch chi tiêu.'
            : 'Not enough expense data to provide suggestions. Please add at least 5 expense transactions.';
    }

    const formattedData = formatTransactionsForPrompt(transactions, categories);
    const langInstruction = language === 'vi' ? 'Hãy trả lời bằng tiếng Việt.' : 'Please respond in English.';

    const prompt = `
        Dựa trên lịch sử giao dịch gần đây, hãy phân tích thói quen chi tiêu của người dùng và đưa ra các mẹo tiết kiệm cụ thể, hữu ích và độc đáo.
        ${langInstruction}
        
        Lịch sử giao dịch:
        ${formattedData}

        Đưa ra 2-3 gạch đầu dòng với các đề xuất có thể thực hiện được. Tập trung vào các lĩnh vực mà họ có thể đang chi tiêu quá nhiều.
    `;

    try {
        // FIX: Updated model name to 'gemini-3-flash-preview' for basic text tasks as per guidelines.
        const response = await ai.models.generateContent({
            model: 'gemini-3-flash-preview',
            contents: prompt,
        });
        // FIX: Ensured string return by providing fallback for potentially undefined text property.
        return response.text || '';
    } catch (error) {
        console.error("Error getting savings suggestion:", error);
        return language === 'vi' ? 'Đã xảy ra lỗi khi kết nối với AI. Vui lòng thử lại sau.' : 'An error occurred while contacting the AI. Please try again later.';
    }
};

export const getSpendingPrediction = async (transactions: Transaction[], categories: Category[], language: string): Promise<string> => {
    if (transactions.filter(t => t.type === 'expense').length < 10) {
        return language === 'vi' 
            ? 'Cần ít nhất 10 giao dịch chi tiêu để đưa ra dự đoán.' 
            : 'At least 10 expense transactions are needed for a prediction.';
    }

    const formattedData = formatTransactionsForPrompt(transactions, categories);
     const langInstruction = language === 'vi' ? 'Hãy trả lời bằng tiếng Việt.' : 'Please respond in English.';

    const prompt = `
        Phân tích lịch sử chi tiêu sau đây và dự đoán tổng chi tiêu cho tháng tới. 
        Đưa ra một con số ước tính và giải thích ngắn gọn lý do cho dự đoán đó dựa trên các xu hướng đã thấy.
        ${langInstruction}

        Lịch sử chi tiêu:
        ${formattedData}
    `;

    try {
        // FIX: Updated model name to 'gemini-3-flash-preview' for basic text tasks.
        const response = await ai.models.generateContent({
            model: 'gemini-3-flash-preview',
            contents: prompt,
        });
        // FIX: Ensured string return by providing fallback for potentially undefined text property.
        return response.text || '';
    } catch (error) {
        console.error("Error getting spending prediction:", error);
        return language === 'vi' ? 'Lỗi dự đoán chi tiêu. Vui lòng thử lại.' : 'Error predicting spending. Please try again.';
    }
};
