import React, { createContext, useContext, useState, useCallback, useRef, useEffect } from 'react';

interface ToastContextType {
  showToast: (message: string) => void;
  toastMessage: string | null;
  isToastVisible: boolean;
}

const ToastContext = createContext<ToastContextType | undefined>(undefined);

export const ToastProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
  const [toastMessage, setToastMessage] = useState<string | null>(null);
  const [isToastVisible, setIsToastVisible] = useState(false);
  const timeoutRef = useRef<number | null>(null);

  const showToast = useCallback((message: string) => {
    if (timeoutRef.current) {
      clearTimeout(timeoutRef.current);
    }
    setToastMessage(message);
    setIsToastVisible(true);
    
    timeoutRef.current = window.setTimeout(() => {
      setIsToastVisible(false);
      setTimeout(() => setToastMessage(null), 500); 
    }, 3000);
  }, []);

  useEffect(() => {
    return () => {
      if (timeoutRef.current) {
        clearTimeout(timeoutRef.current);
      }
    };
  }, []);

  const value = { showToast, toastMessage, isToastVisible };

  return <ToastContext.Provider value={value}>{children}</ToastContext.Provider>;
};

export const useToast = () => {
  const context = useContext(ToastContext);
  if (context === undefined) {
    throw new Error('useToast must be used within a ToastProvider');
  }
  return context;
};
