
import React, { createContext, useContext, useMemo, useEffect, useRef } from 'react';
import { Transaction, Category, Wallet, Debt } from '../types';
import { useLocalStorage } from '../hooks/useLocalStorage';
import { DEFAULT_CATEGORIES, DEFAULT_WALLETS } from '../constants';
import { useLanguage } from './LanguageContext';
import { useToast } from './ToastContext';

interface DataContextType {
    transactions: Transaction[];
    categories: Category[];
    wallets: Wallet[];
    addTransaction: (transaction: Omit<Transaction, 'id'>) => Transaction;
    addTransactionsBulk: (newTransactions: Omit<Transaction, 'id'>[]) => void;
    updateTransaction: (transaction: Transaction) => void;
    deleteTransaction: (id: string) => void;
    addCategory: (category: Omit<Category, 'id' | 'order'>) => Category;
    updateCategory: (category: Category) => void;
    deleteCategory: (id: string) => { success: boolean; message?: string };
    updateAllCategories: (categories: Category[]) => void;
    addWallet: (wallet: Omit<Wallet, 'id' | 'order'>) => void;
    updateWallet: (wallet: Wallet) => void;
    deleteWallet: (id: string) => void;
    updateAllWallets: (wallets: Wallet[]) => void;
    getWalletBalance: (walletId: string) => number;
    totalBalance: number;
    addTransfer: (data: { fromWalletId: string; toWalletId: string; amount: number; date: string; description: string }) => void;
    deleteTransfer: (transferId: string) => void;
    debts: Debt[];
    addDebt: (debt: Omit<Debt, 'id' | 'outstandingAmount' | 'status'>) => void;
    deleteDebt: (id: string) => void;
    addRepayment: (data: { debtId: string; amount: number; date: string; walletId: string }) => void;
    addDebtAndInitialTransaction: (data: { type: 'loan' | 'debt'; person: string; amount: number; date: string; walletId: string; description: string; dueDate?: string; }) => void;
    backupData: () => string;
    restoreData: (data: any) => { success: boolean };
    netTransactionsForReporting: Transaction[];
    transactionsForDisplay: Transaction[];
    repaymentsByTxId: Map<string, Transaction[]>;
}

const DataContext = createContext<DataContextType | undefined>(undefined);

const useUpdateEffect = (effect: React.EffectCallback, deps?: React.DependencyList) => {
    const isInitialMount = useRef(true);
    useEffect(() => {
        if (isInitialMount.current) {
            isInitialMount.current = false;
        } else {
            return effect();
        }
    }, deps);
};

export const DataProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
    const [transactions, setTransactions] = useLocalStorage<Transaction[]>('transactions', []);
    const [categories, setCategories] = useLocalStorage<Category[]>('categories', DEFAULT_CATEGORIES);
    const [wallets, setWallets] = useLocalStorage<Wallet[]>('wallets', DEFAULT_WALLETS);
    const [debts, setDebts] = useLocalStorage<Debt[]>('debts', []);
    const { t } = useLanguage();
    const { showToast } = useToast();

    const stringifiedData = useMemo(() => JSON.stringify({
        transactions,
        categories,
        wallets,
        debts
    }), [transactions, categories, wallets, debts]);

    useUpdateEffect(() => {
        showToast(t('dataSaved'));
    }, [stringifiedData]);

    const repaymentsByTxId = useMemo(() => {
        const map = new Map<string, Transaction[]>();
        transactions
            .filter(tx => tx.isShareBillRepayment && tx.linkedShareBillTxId)
            .forEach(repayment => {
                if (!map.has(repayment.linkedShareBillTxId!)) {
                    map.set(repayment.linkedShareBillTxId!, []);
                }
                map.get(repayment.linkedShareBillTxId!)!.push(repayment);
            });
        return map;
    }, [transactions]);

    const transactionsForDisplay = useMemo(() => {
        return transactions.map(tx => {
            if (tx.isShareBill && tx.type === 'expense' && repaymentsByTxId.has(tx.id)) {
                const totalRepaid = repaymentsByTxId.get(tx.id)!.reduce((sum, repayment) => sum + repayment.amount, 0);
                const netAmount = tx.amount - totalRepaid;
                if (netAmount < 0) {
                    return { ...tx, amount: Math.abs(netAmount), type: 'income' as const };
                }
                return { ...tx, amount: netAmount };
            }
            return tx;
        });
    }, [transactions, repaymentsByTxId]);

    const netTransactionsForReporting = useMemo(() => {
        return transactionsForDisplay.filter(tx => !tx.excludeFromReport);
    }, [transactionsForDisplay]);

    const addTransaction = (transaction: Omit<Transaction, 'id'>) => {
        const newTransaction: Transaction = { ...transaction, id: `tx-${new Date().getTime()}-${Math.random().toString(36).substr(2, 9)}` };
        setTransactions(prev => [...prev, newTransaction].sort((a, b) => {
            const dateComparison = new Date(b.date).getTime() - new Date(a.date).getTime();
            if (dateComparison !== 0) return dateComparison;
            return b.id.localeCompare(a.id);
        }));
        return newTransaction;
    };

    const addTransactionsBulk = (newTransactions: Omit<Transaction, 'id'>[]) => {
        const timestamp = new Date().getTime();
        const processed = newTransactions.map((tx, idx) => ({
            ...tx,
            id: `tx-import-${timestamp}-${idx}`
        }));
        setTransactions(prev => [...prev, ...processed].sort((a, b) => {
            const dateComparison = new Date(b.date).getTime() - new Date(a.date).getTime();
            if (dateComparison !== 0) return dateComparison;
            return b.id.localeCompare(a.id);
        }));
    };

    const updateTransaction = (updatedTransaction: Transaction) => {
        setTransactions(prev => prev.map(t => t.id === updatedTransaction.id ? updatedTransaction : t));
    };

    const deleteTransaction = (id: string) => {
         setTransactions(prev => prev.filter(t => t.id !== id));
    };

    const addCategory = (category: Omit<Category, 'id' | 'order'>): Category => {
        const siblings = categories.filter(c => c.parentId === category.parentId);
        const newOrder = siblings.length > 0 ? Math.max(...siblings.map(s => s.order)) + 1 : 0;
        const newCategory: Category = { ...category, id: `cat-${new Date().getTime()}`, order: newOrder };
        setCategories(prev => [...prev, newCategory]);
        return newCategory;
    };

    const updateCategory = (updatedCategory: Category) => {
        setCategories(prev => prev.map(c => c.id === updatedCategory.id ? updatedCategory : c));
    };
    
    const updateAllCategories = (updatedCategories: Category[]) => {
        setCategories(updatedCategories);
    };

    const deleteCategory = (id: string): { success: boolean, message?: string } => {
        const hasChildren = categories.some(c => c.parentId === id);
        if (hasChildren) return { success: false, message: t('categoryDeleteErrorChildren') };
        const isInUse = transactions.some(t => t.categoryId === id);
        if (isInUse) return { success: false, message: t('categoryDeleteErrorInUse') };
        setCategories(prev => prev.filter(c => c.id !== id));
        return { success: true };
    };
    
    const addWallet = (wallet: Omit<Wallet, 'id' | 'order'>) => {
        const siblings = wallets.filter(w => !w.parentWalletId);
        const newOrder = siblings.length > 0 ? Math.max(...siblings.map(s => s.order)) + 1 : 0;
        const newWallet: Wallet = { ...wallet, id: `wallet-${new Date().getTime()}`, order: newOrder };
        setWallets(prev => [...prev, newWallet]);
    };

    const updateWallet = (updatedWallet: Wallet) => {
        setWallets(prev => prev.map(w => w.id === updatedWallet.id ? updatedWallet : w));
    };

    const updateAllWallets = (updatedWallets: Wallet[]) => {
        setWallets(updatedWallets);
    };

    const deleteWallet = (id: string) => {
        setWallets(prev => prev.filter(w => w.id !== id));
        setTransactions(prev => prev.filter(t => t.walletId !== id));
    };

    const addTransfer = (data: { fromWalletId: string; toWalletId: string; amount: number; date: string; description: string }) => {
        const transferId = `tf-${new Date().getTime()}`;
        const fromWallet = wallets.find(w => w.id === data.fromWalletId);
        const toWallet = wallets.find(w => w.id === data.toWalletId);

        const expenseTx: Omit<Transaction, 'id'> = {
            ...data, type: 'expense', categoryId: 'cat-transfer',
            description: `${data.description || t('transfer')} (${t('transferTo')} ${toWallet?.name})`,
            walletId: data.fromWalletId, transferId: transferId,
        };
        const incomeTx: Omit<Transaction, 'id'> = {
            ...data, type: 'income', categoryId: 'cat-transfer',
            description: `${data.description || t('transfer')} (${t('transferFrom')} ${fromWallet?.name})`,
            walletId: data.toWalletId, transferId: transferId,
        };
        addTransaction(expenseTx);
        addTransaction(incomeTx);
    };

    const deleteTransfer = (transferId: string) => {
        setTransactions(prev => prev.filter(t => t.transferId !== transferId));
    };

    const addDebt = (debt: Omit<Debt, 'id' | 'outstandingAmount' | 'status'>) => {
        const newDebt: Debt = {
            ...debt,
            id: `debt-${new Date().getTime()}`,
            outstandingAmount: debt.initialAmount,
            status: debt.initialAmount > 0 ? 'open' : 'paid',
        };
        setDebts(prev => [...prev, newDebt].sort((a, b) => new Date(b.creationDate).getTime() - new Date(a.creationDate).getTime()));
    };

    const addDebtAndInitialTransaction = (data: {
        type: 'loan' | 'debt';
        person: string;
        amount: number;
        date: string;
        walletId: string;
        description: string;
        dueDate?: string;
    }) => {
        const debtId = `debt-${new Date().getTime()}`;
        const newDebt: Debt = {
            id: debtId,
            type: data.type,
            person: data.person,
            description: data.description,
            initialAmount: data.amount,
            outstandingAmount: data.amount,
            creationDate: data.date,
            dueDate: data.dueDate,
            status: 'open',
        };
        setDebts(prev => [...prev, newDebt].sort((a, b) => new Date(b.creationDate).getTime() - new Date(a.creationDate).getTime()));
        const descriptionForTx = data.description || (data.type === 'loan' ? t('lendTo', {person: data.person}) : t('borrowFrom', {person: data.person}));
        const initialTx: Omit<Transaction, 'id'> = {
            date: data.date,
            description: descriptionForTx,
            amount: data.amount,
            categoryId: 'cat-debt',
            type: data.type === 'loan' ? 'expense' : 'income',
            walletId: data.walletId,
            debtId: debtId,
        };
        addTransaction(initialTx);
    };

    const addRepayment = (data: { debtId: string; amount: number; date: string; walletId: string }) => {
        const debt = debts.find(d => d.id === data.debtId);
        if (!debt) return;
        const newOutstanding = debt.outstandingAmount - data.amount;
        const updatedDebt: Debt = {
            ...debt,
            outstandingAmount: newOutstanding,
            status: newOutstanding <= 0.001 ? 'paid' : 'open',
        };
        setDebts(prev => prev.map(d => d.id === data.debtId ? updatedDebt : d));
        const repaymentTx: Omit<Transaction, 'id'> = {
            date: data.date,
            description: `${t(debt.type === 'loan' ? 'repaymentFrom' : 'repaymentTo')} ${debt.person}`,
            amount: data.amount,
            categoryId: 'cat-debt',
            type: debt.type === 'loan' ? 'income' : 'expense',
            walletId: data.walletId,
            debtId: data.debtId,
        };
        addTransaction(repaymentTx);
    };

    const deleteDebt = (id: string) => {
        setDebts(prev => prev.filter(d => d.id !== id));
        setTransactions(prev => prev.filter(t => t.debtId !== id));
    };
    
    const getWalletBalance = (walletId: string) => {
        const wallet = wallets.find(w => w.id === walletId);
        if (!wallet) return 0;
        return transactions
            .filter(t => t.walletId === walletId)
            .reduce((acc, t) => {
                if (t.isShareBillRepayment) return acc;
                let transactionAmount = t.amount;
                if (t.isShareBill && repaymentsByTxId.has(t.id)) {
                    const totalRepaid = repaymentsByTxId.get(t.id)!
                        .reduce((sum, repayment) => sum + repayment.amount, 0);
                    transactionAmount -= totalRepaid;
                }
                return t.type === 'income' ? acc + transactionAmount : acc - transactionAmount;
            }, wallet.initialBalance);
    };

    const totalBalance = useMemo(() => {
        return wallets.filter(w => w.type !== 'credit').reduce((total, w) => total + getWalletBalance(w.id), 0);
    }, [wallets, transactions, repaymentsByTxId]);
    
    const backupData = () => {
        return JSON.stringify({ transactions, categories, wallets, debts }, null, 2);
    };
    
    const restoreData = (data: any): { success: boolean } => {
        try {
            if (typeof data !== 'object' || data === null) return { success: false };
            const transactionsToRestore = Array.isArray(data.transactions) ? data.transactions : [];
            const categoriesToRestore = Array.isArray(data.categories) ? data.categories : DEFAULT_CATEGORIES;
            const walletsToRestore = Array.isArray(data.wallets) ? data.wallets : DEFAULT_WALLETS;
            const debtsToRestore = Array.isArray(data.debts) ? data.debts : [];
            window.localStorage.setItem('transactions', JSON.stringify(transactionsToRestore));
            window.localStorage.setItem('categories', JSON.stringify(categoriesToRestore));
            window.localStorage.setItem('wallets', JSON.stringify(walletsToRestore));
            window.localStorage.setItem('debts', JSON.stringify(debtsToRestore));
            setTransactions(transactionsToRestore);
            setCategories(categoriesToRestore);
            setWallets(walletsToRestore);
            setDebts(debtsToRestore);
            return { success: true };
        } catch (e) {
            return { success: false };
        }
    };
    
    const value = {
        transactions, categories, wallets, addTransaction, addTransactionsBulk, updateTransaction, deleteTransaction,
        addCategory, updateCategory, deleteCategory, updateAllCategories,
        addWallet, updateWallet, deleteWallet, updateAllWallets,
        getWalletBalance, totalBalance, addTransfer, deleteTransfer,
        debts, addDebt, deleteDebt, addRepayment, addDebtAndInitialTransaction,
        backupData, restoreData, netTransactionsForReporting, transactionsForDisplay, repaymentsByTxId,
    };

    return <DataContext.Provider value={value}>{children}</DataContext.Provider>;
};

export const useData = () => {
    const context = useContext(DataContext);
    if (context === undefined) throw new Error('useData must be used within a DataProvider');
    return context;
};
