
import React, { createContext, useContext, useState, useMemo } from 'react';
import { useLocalStorage } from '../hooks/useLocalStorage';
import { LanguageContent } from '../types';

interface LanguageContextType {
    language: string;
    setLanguage: (lang: string) => void;
    t: (key: string, options?: { [key: string]: string | number } | string) => string;
    formatDate: (dateString: string) => string;
    formatFullDate: (dateString: string) => string;
    parseDate: (displayString: string) => string;
    formatCurrencyInput: (value: string) => string;
    parseCurrencyInput: (value: string) => number;
}

const translations: { [key: string]: LanguageContent } = {
    en: {
      appName: "Sói's Expense Ledger",
      dashboard: 'Dashboard',
      transactions: 'Transactions',
      settings: 'Settings',
      wallets: 'Wallets',
      debtLedger: 'Debt Ledger',
      reports: 'Reports',
      totalBalance: 'Total Balance',
      recentTransactions: 'Recent Transactions',
      getSavingsTip: 'Get AI Savings Tip',
      aiSuggestion: 'AI Suggestion',
      aiThinking: 'AI is analyzing your spending...',
      getSuggestionPrompt: 'Click the button to get personalized savings tips.',
      spendingByTopLevelCategory: 'Spending by Top-Level Category',
      incomeByTopLevelCategory: 'Income by Top-Level Category',
      spendingDistribution: 'Spending Distribution',
      financialCalendar: 'Financial Calendar',
      monthlySummary: 'Monthly Summary',
      addTransaction: 'Add Transaction',
      editTransaction: 'Edit Transaction',
      date: 'Date',
      description: 'Description',
      category: 'Category',
      amount: 'Amount',
      type: 'Type',
      income: 'Income',
      expense: 'Expense',
      actions: 'Actions',
      wallet: 'Wallet',
      noTransactions: 'No transactions yet.',
      filterByDate: 'Filter by Date',
      filterByCategory: 'Filter by Category',
      last10Days: 'Last 10 Days',
      last30Days: 'Last 30 Days',
      thisMonth: 'This Month',
      last3Months: 'Last 3 Months',
      thisYear: 'This Year',
      allTime: 'All Time',
      allCategories: 'All Categories',
      noTransactionsMatchFilter: 'No transactions match your filters.',
      loanDebt: 'Loan/Debt',
      loan: 'Lend',
      debt: 'Borrow',
      lendTo: 'Lent to {person}',
      borrowFrom: 'Borrowed from {person}',
      shareBill: "Share Bill",
      done: "Done",
      isShareBillRepayment: "Receive shared money",
      selectShareBillExpense: "Select expense to offset",
      originalAmount: "from {amount}",
      excludeFromReport: 'Exclude from reports',
      excludeFromReportTooltip: 'Check this for transactions like investment withdrawals or capital returns that should not be counted in income/expense reports.',
      importData: 'Import Data',
      downloadSample: 'Download Sample',
      downloadCategories: 'Download Categories',
      importSuccess: 'Successfully imported {count} transactions!',
      importError: 'Error importing data. Please check your file format.',
      importSummary: 'Import Summary',
      importProcessed: 'Total Processed',
      importSuccessful: 'Successful',
      importDuplicates: 'Duplicates Skipped',
      importFailed: 'Failed',
      importLine: 'Line',
      exportReports: 'Export Reports',
      generalReport: 'General Summary',
      detailedReport: 'Detailed Transactions',
      exportCSV: 'Export as CSV',
      exportJSON: 'Export as JSON',
      reportPeriod: 'Report Period',
      generateReport: 'Generate Report',
      addWallet: 'Add Wallet',
      editWallet: 'Edit Wallet',
      walletName: 'Wallet Name',
      walletType: 'Wallet Type',
      cash: 'Cash',
      bank: 'Bank Account',
      credit: 'Credit Card',
      crypto: 'Cryptocurrency',
      initialBalance: 'Initial Balance',
      currentBalance: 'Current Balance',
      transferFunds: 'Transfer Funds',
      transfer: 'Transfer',
      fromWallet: 'From Wallet',
      toWallet: 'To Wallet',
      transferTo: 'Transfer to',
      transferFrom: 'Transfer from',
      confirmDeleteWallet: 'Are you sure you want to delete this wallet and all its transactions?',
      confirmDeleteTransfer: 'This is a transfer transaction. Deleting it will also delete the corresponding transaction in the other wallet. Are you sure?',
      creditLimit: 'Credit Limit',
      parentWallet: 'Main Card',
      selectParentWallet: 'Select a main card (optional)',
      limitUsage: 'Limit Usage',
      showOnDashboard: 'Show on Dashboard',
      dashboardDisplay: 'Dashboard Display',
      maxWalletsOnDashboardError: 'You can only display a maximum of 4 wallets on the dashboard.',
      walletsCannotBeSame: 'From and To wallets cannot be the same.',
      addLoan: 'Add Loan',
      addDebt: 'Add Debt',
      totalLent: 'Total Lent',
      totalDebt: 'Total Owed',
      loans: 'I Lent Out (Loans)',
      debts: 'I Owe (Debts)',
      noLoans: 'No loans recorded yet.',
      noDebts: 'You have no recorded debts.',
      confirmDeleteDebt: 'Are you sure you want to delete this debt record? This will also delete any associated repayment transactions.',
      outstanding: 'Outstanding',
      dueDate: 'Due Date',
      recordRepayment: 'Record Repayment',
      person: 'Person',
      creationDate: 'Creation Date',
      repaymentTo: 'Repayment to',
      repaymentFrom: 'Repayment from',
      repaymentAmount: 'Repayment Amount',
      repaymentDate: 'Repayment Date',
      status: 'Status',
      open: 'Open',
      paid: 'Paid',
      language: 'Language',
      languagePrompt: 'Choose your preferred language.',
      dataManagement: 'Data Management',
      backupData: 'Backup Data',
      restoreData: 'Restore Data',
      categoryManagement: 'Category Management',
      expenseCategories: 'Expense Categories',
      incomeCategories: 'Income Categories',
      systemCategories: 'System Categories',
      confirmRestore: 'Are you sure you want to restore data? This will overwrite all current data.',
      restoreSuccess: 'Data restored successfully!',
      restoreSuccessTitle: 'Restore Successful',
      restoreError: 'Failed to restore data. The file may be invalid or corrupted.',
      restoreErrorTitle: 'Restore Failed',
      addCategory: 'Add Category',
      editCategory: 'Edit Category',
      categoryName: 'Category Name',
      parentCategory: 'Parent Category',
      noParent: 'No parent (top-level)',
      icon: 'Icon',
      color: 'Color',
      selectIcon: 'Select an icon',
      categoryDeleteError: 'Cannot Delete Category',
      categoryDeleteErrorChildren: 'This category has sub-categories. Please delete or re-assign them first.',
      categoryDeleteErrorInUse: 'This category is being used in transactions. Please re-assign them first.',
      confirmDeleteCategory: 'Are you sure you want to delete this category?',
      save: 'Save',
      cancel: 'Cancel',
      delete: 'Delete',
      confirmDelete: 'Are you sure you want to delete this transaction?',
      selectCategory: 'Select a category',
      selectWallet: 'Select a wallet',
      selectWalletType: 'Select a wallet type',
      allFieldsRequired: 'Please fill out all fields.',
      validAmount: 'Please enter a valid positive amount.',
      optional: 'optional',
      ok: 'OK',
      dataSaved: 'Data saved automatically.',
      updateReminder: "It's been {days} days since your last update. Don't forget to record your expenses!",
      showDailyStats: "Show Daily Stats",
      selectWalletsForStats: "Select wallets for balance (max 2)",
      dayIncome: "THU",
      dayExpense: "CHI",
      dailyBalance: "Số dư cuối ngày",
      addImage: "Add Photo",
      removeImage: "Remove Photo",
      categoryComparison: "Category Comparison",
      selectCategories: "Select Categories to Compare",
      allExpenses: "All Expenses",
      customRange: "Custom",
      startDate: "From",
      endDate: "To"
    },
    vi: {
      appName: 'Sổ chi tiêu của Sói',
      dashboard: 'Bảng điều khiển',
      transactions: 'Giao dịch',
      settings: 'Cài đặt',
      wallets: 'Các loại ví',
      debtLedger: 'Sổ nợ',
      reports: 'Báo cáo',
      totalBalance: 'Tổng số dư',
      recentTransactions: 'Giao dịch Gần đây',
      getSavingsTip: 'Nhận Mẹo Tiết kiệm AI',
      aiSuggestion: 'Gợi ý từ AI',
      aiThinking: 'AI đang phân tích chi tiêu của bạn...',
      getSuggestionPrompt: 'Nhấn nút để nhận mẹo tiết kiệm được cá nhân hóa.',
      spendingByTopLevelCategory: 'Chi tiêu theo Danh mục chính',
      incomeByTopLevelCategory: 'Thu nhập theo Danh mục chính',
      spendingDistribution: 'Phân bổ Chi tiêu',
      financialCalendar: 'Lịch tài chính',
      monthlySummary: 'Tổng quan hàng tháng',
      addTransaction: 'Thêm Giao dịch',
      editTransaction: 'Sửa Giao dịch',
      date: 'Ngày',
      description: 'Mô tả',
      category: 'Phân loại',
      amount: 'Số tiền',
      type: 'Loại',
      income: 'Thu nhập',
      expense: 'Chi tiêu',
      actions: 'Hành động',
      wallet: 'Ví',
      noTransactions: 'Chưa có giao dịch nào.',
      filterByDate: 'Lọc theo ngày',
      filterByCategory: 'Lọc theo phân loại',
      last10Days: '10 ngày qua',
      last30Days: '30 ngày qua',
      thisMonth: 'Tháng này',
      last3Months: '3 tháng qua',
      thisYear: 'Trong năm nay',
      allTime: 'Tất cả thời gian',
      allCategories: 'Tất cả phân loại',
      noTransactionsMatchFilter: 'Không có giao dịch nào khớp với bộ lọc của bạn.',
      loanDebt: 'Vay/Nợ',
      loan: 'Cho vay',
      debt: 'Đi vay',
      lendTo: 'Cho {person} vay',
      borrowFrom: 'Vay từ {person}',
      shareBill: "Chia hóa đơn",
      done: "Hoàn tất",
      isShareBillRepayment: "Nhận tiền share bill",
      selectShareBillExpense: "Chọn giao dịch chi cần đối trừ",
      originalAmount: "từ {amount}",
      excludeFromReport: 'Không tính vào báo cáo',
      excludeFromReportTooltip: 'Dành cho các giao dịch như rút vốn đầu tư, không nên tính vào báo cáo thu/chi.',
      importData: 'Nhập dữ liệu',
      downloadSample: 'Tải tệp mẫu',
      downloadCategories: 'Tải danh sách phân loại',
      importSuccess: 'Đã nhập thành công {count} giao dịch!',
      importError: 'Lỗi khi nhập dữ liệu. Vui lòng kiểm tra lại định dạng tệp.',
      importSummary: 'Tổng kết Nhập dữ liệu',
      importProcessed: 'Tổng số dòng xử lý',
      importSuccessful: 'Thành công',
      importDuplicates: 'Bỏ qua (Trùng lặp)',
      importFailed: 'Thất bại (Lỗi)',
      importLine: 'Dòng',
      exportReports: 'Xuất báo cáo',
      generalReport: 'Báo cáo tổng quát',
      detailedReport: 'Báo cáo chi tiết',
      exportCSV: 'Xuất CSV (Excel)',
      exportJSON: 'Xuất JSON',
      reportPeriod: 'Khoảng thời gian',
      generateReport: 'Tạo báo cáo',
      addWallet: 'Thêm ví',
      editWallet: 'Sửa ví',
      walletName: 'Tên ví',
      walletType: 'Loại ví',
      cash: 'Tiền mặt',
      bank: 'Tài khoản ngân hàng',
      credit: 'Thẻ tín dụng',
      crypto: 'Tiền ảo',
      initialBalance: 'Số dư ban đầu',
      currentBalance: 'Số dư hiện tại',
      transferFunds: 'Chuyển tiền',
      transfer: 'Chuyển tiền',
      fromWallet: 'Từ ví',
      toWallet: 'Đến ví',
      transferTo: 'Chuyển đến',
      transferFrom: 'Chuyển từ',
      confirmDeleteWallet: 'Bạn có chắc muốn xóa ví này và tất cả giao dịch của nó không?',
      confirmDeleteTransfer: 'Đây là giao dịch chuyển tiền. Xóa nó sẽ xóa cả giao dịch tương ứng ở ví kia. Bạn chắc chắn chứ?',
      creditLimit: 'Hạn mức tín dụng',
      parentWallet: 'Thẻ chính',
      selectParentWallet: 'Chọn thẻ chính (không bắt buộc)',
      limitUsage: 'Sử dụng Hạn mức',
      showOnDashboard: 'Hiển thị trên Bảng điều khiển',
      dashboardDisplay: 'Hiển thị',
      maxWalletsOnDashboardError: 'Bạn chỉ có thể hiển thị tối đa 4 ví trên bảng điều khiển.',
      walletsCannotBeSame: 'Ví gửi và ví nhận không được trùng nhau.',
      addLoan: 'Thêm Khoản cho vay',
      addDebt: 'Thêm Khoản nợ',
      totalLent: 'Tổng tiền cho vay',
      totalDebt: 'Tổng tiền nợ',
      loans: 'Khoản cho vay',
      debts: 'Khoản nợ',
      noLoans: 'Chưa có khoản cho vay nào.',
      noDebts: 'Bạn không có khoản nợ nào.',
      confirmDeleteDebt: 'Bạn có chắc muốn xóa khoản nợ này không? Hành động này cũng sẽ xóa các giao dịch trả nợ liên quan.',
      outstanding: 'Còn lại',
      dueDate: 'Ngày đáo hạn',
      recordRepayment: 'Ghi nhận Trả nợ',
      person: 'Người liên quan',
      creationDate: 'Ngày tạo',
      repaymentTo: 'Trả nợ cho',
      repaymentFrom: 'Nhận tiền trả nợ từ',
      repaymentAmount: 'Số tiền trả',
      repaymentDate: 'Ngày trả',
      status: 'Trạng thái',
      open: 'Chưa trả',
      paid: 'Đã trả',
      language: 'Ngôn ngữ',
      languagePrompt: 'Chọn ngôn ngữ bạn muốn sử dụng.',
      dataManagement: 'Quản lý Dữ liệu',
      backupData: 'Sao lưu Dữ liệu',
      restoreData: 'Phục hồi Dữ liệu',
      categoryManagement: 'Quản lý Phân loại',
      expenseCategories: 'Phân loại Chi tiêu',
      incomeCategories: 'Phân loại Thu nhập',
      systemCategories: 'Phân loại Hệ thống',
      confirmRestore: 'Bạn có chắc chắn muốn phục hồi dữ liệu không? Hành động này sẽ ghi đè toàn bộ dữ liệu hiện tại.',
      restoreSuccess: 'Phục hồi dữ liệu thành công!',
      restoreSuccessTitle: 'Phục hồi Thành công',
      restoreError: 'Phục hồi dữ liệu thất bại. Tệp có thể không hợp lệ hoặc đã bị hỏng.',
      restoreErrorTitle: 'Phục hồi Thất bại',
      addCategory: 'Thêm Phân loại',
      editCategory: 'Sửa Phân loại',
      categoryName: 'Tên Phân loại',
      parentCategory: 'Phân loại cha',
      noParent: 'Không có (cấp cao nhất)',
      icon: 'Biểu tượng',
      color: 'Màu sắc',
      selectIcon: 'Chọn một biểu tượng',
      categoryDeleteError: 'Không thể xóa Phân loại',
      categoryDeleteErrorChildren: 'Phân loại này có các mục con. Vui lòng xóa hoặc di chuyển chúng trước.',
      categoryDeleteErrorInUse: 'Phân loại này đang được sử dụng trong các giao dịch. Vui lòng thay đổi chúng trước.',
      confirmDeleteCategory: 'Bạn có chắc muốn xóa phân loại này không?',
      save: 'Lưu',
      cancel: 'Hủy',
      delete: 'Xóa',
      confirmDelete: 'Bạn có chắc chắn muốn xóa giao dịch này không?',
      selectCategory: 'Chọn một phân loại',
      selectWallet: 'Chọn một ví',
      selectWalletType: 'Chọn một loại ví',
      allFieldsRequired: 'Vui lòng điền đầy đủ thông tin.',
      validAmount: 'Vui lòng nhập số tiền hợp lệ.',
      optional: 'không bắt buộc',
      ok: 'OK',
      dataSaved: 'Đã tự động lưu dữ liệu.',
      updateReminder: "Đã {days} ngày bạn chưa cập nhật chi tiêu. Đừng quên ghi lại các giao dịch nhé!",
      showDailyStats: "Hiển thị thống kê ngày",
      selectWalletsForStats: "Chọn ví xem số dư (tối đa 2)",
      dayIncome: "THU",
      dayExpense: "CHI",
      dailyBalance: "Số dư cuối ngày",
      addImage: "Thêm ảnh",
      removeImage: "Xóa ảnh",
      categoryComparison: "So sánh các danh mục",
      selectCategories: "Chọn các danh mục để so sánh",
      allExpenses: "Tất cả chi tiêu",
      customRange: "Tùy chỉnh",
      startDate: "Từ ngày",
      endDate: "Đến ngày"
    }
};

const LanguageContext = createContext<LanguageContextType | undefined>(undefined);

export const LanguageProvider: React.FC<{ children: React.ReactNode }> = ({ children }) => {
    const [language, setLanguage] = useLocalStorage<string>('language', 'vi');

    const t = (key: string, options?: { [key: string]: string | number } | string): string => {
        const keys = key.split('.');
        
        const getTranslation = (lang: string): string | undefined => {
            let current: any = translations[lang as keyof typeof translations];
            if (!current) return undefined;

            for (const k of keys) {
                if (current && typeof current === 'object' && k in current) {
                    current = current[k];
                } else {
                    return undefined;
                }
            }
            return typeof current === 'string' ? current : undefined;
        };
        
        let translation = getTranslation(language) ?? getTranslation('en');

        if (translation === undefined) {
            if (typeof options === 'string') {
                return options;
            }
            return key;
        }

        if (typeof options === 'object' && options !== null) {
            Object.entries(options).forEach(([placeholder, value]) => {
                translation = translation!.replace(`{${placeholder}}`, String(value));
            });
        }
        
        return translation;
    };
    
    const formatDate = (dateString: string): string => {
        if (!dateString || !/^\d{4}-\d{2}-\d{2}$/.test(dateString)) return '';
        const [year, month, day] = dateString.split('-');
        return `${day}/${month}/${year}`;
    };

     const formatFullDate = (dateString: string): string => {
        if (!dateString || !/^\d{4}-\d{2}-\d{2}$/.test(dateString)) return '';
        const date = new Date(`${dateString}T00:00:00`);
        const options: Intl.DateTimeFormatOptions = { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' };
        return new Intl.DateTimeFormat(language === 'vi' ? 'vi-VN' : 'en-US', options).format(date);
    };

    const parseDate = (displayString: string): string => {
        if (!displayString) return '';
        const parts = displayString.match(/^(\d{1,2})[/\-.](\d{1,2})[/\-.](\d{4})$/);
        if (!parts) return '';
        const day = parseInt(parts[1], 10);
        const month = parseInt(parts[2], 10);
        const year = parseInt(parts[3], 10);
        
        if (day < 1 || day > 31 || month < 1 || month > 12 || year < 1900) {
            return '';
        }

        const isoString = `${year}-${String(month).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
        const d = new Date(isoString);
        if (d.getFullYear() === year && d.getMonth() + 1 === month && d.getDate() === day) {
            return isoString;
        }
        return '';
    };
    
    const formatCurrencyInput = (value: string): string => {
        if (!value) return '';
        const numberValue = parseInt(value.replace(/\D/g, ''), 10);
        if (isNaN(numberValue)) return '';
        return new Intl.NumberFormat('vi-VN').format(numberValue);
    };

    const parseCurrencyInput = (value: string): number => {
        if (!value) return 0;
        return parseInt(value.replace(/\D/g, ''), 10) || 0;
    };

    const value = { language, setLanguage, t, formatDate, parseDate, formatFullDate, formatCurrencyInput, parseCurrencyInput };

    return <LanguageContext.Provider value={value}>{children}</LanguageContext.Provider>;
};

export const useLanguage = () => {
    const context = useContext(LanguageContext);
    if (context === undefined) {
        throw new Error('useLanguage must be used within a LanguageProvider');
    }
    return context;
};
