
import React, { useState, useMemo, useCallback, useRef } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Transaction, Category, Wallet } from '../types';
import TransactionModal from './TransactionModal';
import { PlusIcon, PencilSquareIcon, TrashIcon, ArrowsRightLeftIcon, Icon, EyeSlashIcon, ArrowDownTrayIcon, ArrowUpTrayIcon, BookOpenIcon, XMarkIcon, Cog6ToothIcon, PhotoIcon } from './icons/Icons';
import { useToast } from '../context/ToastContext';
import { useLocalStorage } from '../hooks/useLocalStorage';

interface ImportSummary {
    total: number;
    success: number;
    duplicates: { date: string, desc: string, amount: number }[];
    errors: { line: number, content: string, reason: string }[];
}

const Transactions: React.FC = () => {
    const { transactions, categories, deleteTransaction, deleteTransfer, wallets, transactionsForDisplay, addTransactionsBulk } = useData();
    const { t, language, formatFullDate } = useLanguage();
    const { showToast } = useToast();
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [editingTransaction, setEditingTransaction] = useState<Transaction | null>(null);
    const fileInputRef = useRef<HTMLInputElement>(null);

    const [dateFilter, setDateFilter] = useState('thisMonth');
    const [categoryFilter, setCategoryFilter] = useState('all');

    // UI State for Stats Settings
    const [showStatsOptions, setShowStatsOptions] = useState(false);
    const [showDailyStats, setShowDailyStats] = useLocalStorage<boolean>('showDailyStats', true);
    const [statsWallets, setStatsWallets] = useLocalStorage<string[]>('statsWallets', []);

    const [importSummary, setImportSummary] = useState<ImportSummary | null>(null);
    const [selectedImage, setSelectedImage] = useState<string | null>(null);

    const cashWalletIds = useMemo(() => {
        return new Set(wallets.filter(w => w.type === 'cash').map(w => w.id));
    }, [wallets]);

    const smartParseDate = (dateStr: string): string => {
        if (!dateStr) return '';
        const parts = dateStr.split(/[/\-.]/);
        if (parts.length !== 3) return '';

        let p1 = parseInt(parts[0], 10);
        let p2 = parseInt(parts[1], 10);
        let p3 = parseInt(parts[2], 10);

        if (isNaN(p1) || isNaN(p2) || isNaN(p3)) return '';

        let year = p3 < 100 ? 2000 + p3 : p3;
        let month, day;

        if (p1 > 12) { 
            day = p1;
            month = p2;
        } else if (p2 > 12) {
            month = p1;
            day = p2;
        } else {
            month = p1;
            day = p2;
        }

        const isoDate = `${year}-${String(month).padStart(2, '0')}-${String(day).padStart(2, '0')}`;
        const d = new Date(isoDate);
        if (d.getFullYear() === year && d.getMonth() + 1 === month && d.getDate() === day) {
            return isoDate;
        }
        return '';
    };

    const handleAddClick = () => {
        setEditingTransaction(null);
        setIsModalOpen(true);
    };

    const handleEditClick = (tx: Transaction) => {
        setEditingTransaction(tx);
        setIsModalOpen(true);
    };

    const handleDeleteClick = (tx: Transaction) => {
        if (tx.transferId) {
            if (window.confirm(t('confirmDeleteTransfer'))) deleteTransfer(tx.transferId);
        } else {
            if (window.confirm(t('confirmDelete'))) deleteTransaction(tx.id);
        }
    };
    
    const formatFullCurrency = (amount: number) => {
        return new Intl.NumberFormat('vi-VN').format(amount) + ' đ';
    }

    const formatCurrency = (amount: number, compact = false) => {
        return new Intl.NumberFormat(language === 'vi' ? 'vi-VN' : 'en-US', {
            style: 'currency', 
            currency: language === 'vi' ? 'VND' : 'USD',
            notation: compact ? 'compact' : 'standard'
        }).format(amount);
    }

    const downloadSample = () => {
        const headers = 'Ngày,Mô tả,Số tiền,Loại(income/expense),Danh mục,Ví';
        const sampleRows = '10/26/2025,Mua lót nồi (Shopee),"135,432",expense,Đồ gia dụng,Tài khoản MB\n10/26/2025,Ăn ốc với cả nhà,340000,expense,Nhà hàng,Tài khoản MB';
        const blob = new Blob([`\ufeff${headers}\n${sampleRows}`], { type: 'text/csv;charset=utf-8;' });
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.setAttribute('download', `wolf_ledger_sample.csv`);
        link.click();
    };

    const downloadCategories = () => {
        const headers = language === 'vi' ? 'Tên phân loại,Phân loại cha' : 'Category Name,Parent Category';
        const rows = categories
            .filter(c => c.id !== 'cat-transfer' && c.id !== 'cat-debt')
            .map(c => {
                const parentName = c.parentId ? categories.find(p => p.id === c.parentId)?.name || '' : '';
                return `"${c.name}","${parentName}"`;
            });
        const blob = new Blob([`\ufeff${headers}\n${rows.join('\n')}`], { type: 'text/csv;charset=utf-8;' });
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.setAttribute('download', `wolf_ledger_categories.csv`);
        link.click();
    };

    const handleImport = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (!file) return;

        const reader = new FileReader();
        reader.onload = (event) => {
            try {
                const text = event.target?.result as string;
                const lines = text.split(/\r?\n/).filter(line => line.trim() !== '');
                if (lines.length < 2) throw new Error('File rỗng');

                const summary: ImportSummary = { total: lines.length - 1, success: 0, duplicates: [], errors: [] };
                const newTxs: Omit<Transaction, 'id'>[] = [];

                for (let i = 1; i < lines.length; i++) {
                    const line = lines[i];
                    const regex = /,(?=(?:(?:[^"]*"){2})*[^"]*$)/;
                    let cols = line.split(regex).map(c => c.trim().replace(/^"|"$/g, ''));

                    if (cols.length === 7) {
                        const amountPart1 = cols[2].replace(/[^0-9]/g, '');
                        const amountPart2 = cols[3].replace(/[^0-9]/g, '');
                        if (/^\d+$/.test(amountPart1) && /^\d+$/.test(amountPart2)) {
                            cols = [cols[0], cols[1], amountPart1 + amountPart2, cols[4], cols[5], cols[6]];
                        }
                    }

                    if (cols.length < 4) {
                        summary.errors.push({ line: i + 1, content: line, reason: 'Dòng không đủ cột thông tin' });
                        continue;
                    }

                    const [rawDate, desc, rawAmount, rawType, catName, walletName] = cols;
                    const date = smartParseDate(rawDate);
                    if (!date) {
                        summary.errors.push({ line: i + 1, content: line, reason: 'Ngày không hợp lệ (Cần mm/dd/yyyy)' });
                        continue;
                    }

                    const amountStr = rawAmount.replace(/[^0-9.]/g, '');
                    const amount = parseFloat(amountStr);
                    if (isNaN(amount)) {
                        summary.errors.push({ line: i + 1, content: line, reason: 'Số tiền không phải là số' });
                        continue;
                    }

                    const isDuplicate = transactions.some(t => 
                        t.date === date && 
                        t.description.trim().toLowerCase() === desc.trim().toLowerCase() && 
                        Math.abs(t.amount - amount) < 1
                    );

                    if (isDuplicate) {
                        summary.duplicates.push({ date, desc, amount });
                        continue;
                    }

                    const type = (rawType || 'expense').toLowerCase().includes('income') ? 'income' : 'expense';
                    
                    let categoryId = categories.find(c => 
                        c.name.trim().toLowerCase() === (catName || '').trim().toLowerCase()
                    )?.id;
                    
                    if (!categoryId) {
                        categoryId = type === 'income' ? 'cat-i4' : 'cat-e17'; 
                    }

                    let walletId = wallets.find(w => 
                        w.name.trim().toLowerCase() === (walletName || '').trim().toLowerCase()
                    )?.id;
                    
                    if (!walletId) {
                        walletId = wallets[0]?.id;
                    }

                    newTxs.push({
                        date,
                        description: desc,
                        amount,
                        type,
                        categoryId,
                        walletId,
                        excludeFromReport: false
                    });
                    summary.success++;
                }

                if (newTxs.length > 0) {
                    addTransactionsBulk(newTxs);
                }
                setImportSummary(summary);
            } catch (err) {
                console.error(err);
                alert(t('importError'));
            }
        };
        reader.readAsText(file);
        if (fileInputRef.current) fileInputRef.current.value = '';
    };

    const childMap = useMemo(() => {
        const map = new Map<string, string[]>();
        categories.forEach(c => {
            if (c.parentId) {
                if (!map.has(c.parentId)) map.set(c.parentId, []);
                map.get(c.parentId)!.push(c.id);
            }
        });
        return map;
    }, [categories]);

    const getAllDescendantIds = useCallback((parentId: string): string[] => {
        const descendants: string[] = [];
        const queue: string[] = [...(childMap.get(parentId) || [])];
        while (queue.length > 0) {
            const currentId = queue.shift()!;
            descendants.push(currentId);
            const children = childMap.get(currentId);
            if (children) queue.push(...children);
        }
        return descendants;
    }, [childMap]);

    const filteredTransactions = useMemo(() => {
        const selectedCategoryIds = categoryFilter === 'all' ? null : [categoryFilter, ...getAllDescendantIds(categoryFilter)];
        const now = new Date();
        let startDate = new Date('1900-01-01');
        switch(dateFilter) {
            case '10d': startDate = new Date(); startDate.setDate(now.getDate() - 10); break;
            case 'thisMonth': startDate = new Date(now.getFullYear(), now.getMonth(), 1); break;
            case '3m': startDate = new Date(); startDate.setMonth(now.getMonth() - 3); break;
            case 'thisYear': startDate = new Date(now.getFullYear(), 0, 1); break;
        }
        startDate.setHours(0, 0, 0, 0);
        return transactionsForDisplay.filter(tx => {
            const txDate = new Date(`${tx.date}T00:00:00`);
            const dateMatch = dateFilter === 'all' || txDate >= startDate;
            const categoryMatch = !selectedCategoryIds || selectedCategoryIds.includes(tx.categoryId);
            return dateMatch && categoryMatch;
        });
    }, [transactionsForDisplay, dateFilter, categoryFilter, getAllDescendantIds]);
    
    const groupedTransactions = useMemo<Record<string, Transaction[]>>(() => {
        return filteredTransactions.reduce<Record<string, Transaction[]>>((acc, tx) => {
            const dateKey = tx.date;
            if (!acc[dateKey]) acc[dateKey] = [];
            acc[dateKey].push(tx);
            return acc;
        }, {} as Record<string, Transaction[]>);
    }, [filteredTransactions]);

    const handleWalletToggle = (id: string) => {
        if (statsWallets.includes(id)) {
            setStatsWallets(statsWallets.filter(w => w !== id));
        } else {
            if (statsWallets.length >= 2) {
                setStatsWallets([statsWallets[1], id]);
            } else {
                setStatsWallets([...statsWallets, id]);
            }
        }
    };

    const getDailySummary = (dateKey: string, txsOnDate: Transaction[]) => {
        const dailyIncome = txsOnDate.filter(t => t.type === 'income').reduce((s, t) => s + t.amount, 0);
        const dailyExpense = txsOnDate.filter(t => t.type === 'expense').reduce((s, t) => s + t.amount, 0);
        
        const balances = statsWallets.map(wid => {
            const wallet = wallets.find(w => w.id === wid);
            if (!wallet) return null;
            // FIXED: Use logic consistent with DataContext's getWalletBalance
            // We must filter out isShareBillRepayment because those are already netted in transactionsForDisplay
            const sumTxs = transactionsForDisplay
                .filter(t => t.walletId === wid && t.date <= dateKey && !t.isShareBillRepayment)
                .reduce((acc, t) => t.type === 'income' ? acc + t.amount : acc - t.amount, 0);
            return { name: wallet.name, balance: wallet.initialBalance + sumTxs };
        }).filter(b => b !== null);

        return { dailyIncome, dailyExpense, balances };
    };

    return (
        <div className="space-y-6">
            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
                <h1 className="text-3xl font-bold text-on-surface">{t('transactions')}</h1>
                <div className="flex flex-wrap gap-2">
                    <button onClick={() => setShowStatsOptions(!showStatsOptions)} className={`flex items-center gap-2 px-3 py-2 rounded-lg transition-colors text-sm ${showStatsOptions ? 'bg-primary text-white' : 'bg-gray-100 text-on-surface-secondary hover:bg-gray-200'}`}>
                        <Cog6ToothIcon className="w-4 h-4" />
                    </button>
                    <button onClick={downloadCategories} className="flex items-center gap-2 bg-gray-100 text-on-surface-secondary px-3 py-2 rounded-lg hover:bg-gray-200 transition-colors text-sm">
                        <BookOpenIcon className="w-4 h-4" />
                        <span>{t('downloadCategories')}</span>
                    </button>
                    <button onClick={downloadSample} className="flex items-center gap-2 bg-gray-100 text-on-surface-secondary px-3 py-2 rounded-lg hover:bg-gray-200 transition-colors text-sm">
                        <ArrowDownTrayIcon className="w-4 h-4" />
                        <span>{t('downloadSample')}</span>
                    </button>
                    <button onClick={() => fileInputRef.current?.click()} className="flex items-center gap-2 bg-secondary text-white px-3 py-2 rounded-lg hover:bg-green-600 transition-colors text-sm">
                        <ArrowUpTrayIcon className="w-4 h-4" />
                        <span>{t('importData')}</span>
                    </button>
                    <input type="file" ref={fileInputRef} className="hidden" accept=".csv" onChange={handleImport} />
                    <button onClick={handleAddClick} className="flex items-center gap-2 bg-primary text-white px-4 py-2 rounded-lg hover:bg-primary-dark transition-colors">
                        <PlusIcon className="w-5 h-5" />
                        <span>{t('addTransaction')}</span>
                    </button>
                </div>
            </div>

            {/* Stats Setting Panel */}
            {showStatsOptions && (
                <div className="bg-surface p-4 rounded-xl shadow-md border border-primary/20 space-y-4 animate-in slide-in-from-top duration-300">
                    <div className="flex items-center gap-3 border-b pb-3">
                        <h3 className="font-bold text-primary">{t('showDailyStats')}</h3>
                        <input 
                            type="checkbox" 
                            checked={showDailyStats} 
                            onChange={e => setShowDailyStats(e.target.checked)}
                            className="h-5 w-5 rounded border-gray-300 text-primary focus:ring-primary cursor-pointer"
                        />
                    </div>
                    {showDailyStats && (
                        <div>
                            <p className="text-sm font-medium mb-2 text-on-surface-secondary">{t('selectWalletsForStats')}</p>
                            <div className="flex flex-wrap gap-2">
                                {wallets.map(w => (
                                    <button 
                                        key={w.id} 
                                        onClick={() => handleWalletToggle(w.id)}
                                        className={`px-3 py-1 rounded-full text-xs font-semibold border transition-all ${statsWallets.includes(w.id) ? 'bg-primary text-white border-primary shadow-sm' : 'bg-white text-on-surface-secondary border-gray-300 hover:border-primary'}`}
                                    >
                                        {w.name}
                                    </button>
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            )}

            {importSummary && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-[60] p-4">
                    <div className="bg-surface rounded-2xl shadow-2xl w-full max-w-2xl max-h-[90vh] flex flex-col">
                        <div className="p-6 border-b flex justify-between items-center bg-primary text-white rounded-t-2xl">
                            <h2 className="text-xl font-bold">{t('importSummary')}</h2>
                            <button onClick={() => setImportSummary(null)} className="hover:bg-white/20 p-1 rounded-lg">
                                <XMarkIcon className="w-6 h-6" />
                            </button>
                        </div>
                        <div className="p-6 overflow-y-auto space-y-6">
                            <div className="grid grid-cols-2 sm:grid-cols-4 gap-4">
                                <div className="p-3 bg-gray-100 rounded-xl text-center border">
                                    <p className="text-xs text-on-surface-secondary mb-1">{t('importProcessed')}</p>
                                    <p className="text-xl font-bold">{importSummary.total}</p>
                                </div>
                                <div className="p-3 bg-green-50 rounded-xl text-center border border-green-200">
                                    <p className="text-xs text-green-700 mb-1">{t('importSuccessful')}</p>
                                    <p className="text-xl font-bold text-green-600">{importSummary.success}</p>
                                </div>
                                <div className="p-3 bg-orange-50 rounded-xl text-center border border-orange-200">
                                    <p className="text-xs text-orange-700 mb-1">{t('importDuplicates')}</p>
                                    <p className="text-xl font-bold text-orange-600">{importSummary.duplicates.length}</p>
                                </div>
                                <div className="p-3 bg-red-50 rounded-xl text-center border border-red-200">
                                    <p className="text-xs text-red-700 mb-1">{t('importFailed')}</p>
                                    <p className="text-xl font-bold text-red-600">{importSummary.errors.length}</p>
                                </div>
                            </div>
                        </div>
                        <div className="p-6 border-t bg-gray-50 rounded-b-2xl flex justify-end">
                            <button onClick={() => setImportSummary(null)} className="px-6 py-2 bg-primary text-white rounded-xl font-bold hover:bg-primary-dark transition-colors shadow-lg">
                                {t('ok')}
                            </button>
                        </div>
                    </div>
                </div>
            )}

            {/* Full Image Modal */}
            {selectedImage && (
                <div className="fixed inset-0 bg-black/90 flex items-center justify-center z-[70] p-4" onClick={() => setSelectedImage(null)}>
                    <img src={selectedImage} alt="Large Attachment" className="max-w-full max-h-full rounded-lg shadow-2xl object-contain animate-in zoom-in-95 duration-200" />
                    <button className="absolute top-6 right-6 text-white bg-white/10 p-2 rounded-full hover:bg-white/20">
                        <XMarkIcon className="w-8 h-8" />
                    </button>
                </div>
            )}

            <div className="bg-surface p-4 rounded-xl shadow-sm flex flex-col sm:flex-row gap-4">
                <div className="flex-1">
                    <label className="block text-sm font-medium text-on-surface-secondary mb-1">{t('filterByDate')}</label>
                    <select value={dateFilter} onChange={e => setDateFilter(e.target.value)} className="block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                        <option value="10d">{t('last10Days')}</option>
                        <option value="thisMonth">{t('thisMonth')}</option>
                        <option value="3m">{t('last3Months')}</option>
                        <option value="thisYear">{t('thisYear')}</option>
                        <option value="all">{t('allTime')}</option>
                    </select>
                </div>
                <div className="flex-1">
                    <label className="block text-sm font-medium text-on-surface-secondary mb-1">{t('filterByCategory')}</label>
                    <select value={categoryFilter} onChange={e => setCategoryFilter(e.target.value)} className="block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                        <option value="all">{t('allCategories')}</option>
                        {categories.filter(c => c.id !== 'cat-transfer').map(c => <option key={c.id} value={c.id}>{c.name}</option>)}
                    </select>
                </div>
            </div>

            <div className="space-y-4">
                {Object.keys(groupedTransactions).length > 0 ? (
                    Object.entries(groupedTransactions as Record<string, Transaction[]>)
                        .sort(([dateA], [dateB]) => dateB.localeCompare(dateA))
                        .map(([date, txsOnDate]) => {
                            const summary = getDailySummary(date, txsOnDate);
                            return (
                                <div key={date} className="mb-6">
                                    <h2 className="text-xl font-semibold text-gray-500 py-3 border-b border-gray-100 sticky top-0 bg-background/90 backdrop-blur-md z-10 px-2">
                                        <span>{formatFullDate(date)}</span>
                                    </h2>
                                    
                                    {showDailyStats && (
                                        <div className="bg-white/60 border-b border-gray-100 flex flex-wrap items-center gap-x-4 py-2 px-4 text-[11px] font-bold text-gray-400 uppercase tracking-tight">
                                            <div className="flex items-center gap-1.5">
                                                <span className="text-green-600">{t('dayIncome')}:</span>
                                                <span className="text-green-700 underline underline-offset-4 decoration-green-200 decoration-1">
                                                    {formatFullCurrency(summary.dailyIncome)}
                                                </span>
                                            </div>
                                            <div className="w-px h-3 bg-gray-200 hidden sm:block mx-1" />
                                            <div className="flex items-center gap-1.5">
                                                <span className="text-red-500">{t('dayExpense')}:</span>
                                                <span className="text-red-700 underline underline-offset-4 decoration-red-200 decoration-1">
                                                    {formatFullCurrency(summary.dailyExpense)}
                                                </span>
                                            </div>
                                            {summary.balances.map((b, idx) => (
                                                <React.Fragment key={b.name}>
                                                    <div className="w-px h-3 bg-gray-200 hidden sm:block mx-1" />
                                                    <div className="flex items-center gap-1.5">
                                                        <span>{b.name.toUpperCase()}:</span>
                                                        <span className="text-blue-600 underline underline-offset-4 decoration-blue-200 decoration-1">
                                                            {formatFullCurrency(b.balance)}
                                                        </span>
                                                    </div>
                                                </React.Fragment>
                                            ))}
                                        </div>
                                    )}

                                    <div className="bg-surface rounded-b-xl shadow-sm divide-y divide-gray-100 border border-gray-100 overflow-hidden">
                                        {txsOnDate.map(tx => {
                                            const category = categories.find(c => c.id === tx.categoryId);
                                            const originalTx = transactions.find(orig => orig.id === tx.id);
                                            const isCashTransaction = cashWalletIds.has(tx.walletId);
                                            return (
                                                <div key={tx.id} className={`flex items-center p-3 transition-colors ${isCashTransaction ? 'bg-gray-50/50 hover:bg-gray-100' : 'hover:bg-gray-50'}`}>
                                                    <div className="flex-shrink-0 flex items-center gap-3 w-1/3">
                                                        {category && (
                                                            <span className="w-9 h-9 rounded-full flex items-center justify-center flex-shrink-0" style={{ backgroundColor: category.color + '20' }}>
                                                                <Icon name={category.icon} className="w-5 h-5" style={{ color: category.color }} />
                                                            </span>
                                                        )}
                                                        <div className="overflow-hidden">
                                                            <p className="font-semibold text-sm text-on-surface truncate">{category?.name}</p>
                                                            <p className="text-[11px] text-on-surface-secondary truncate">{wallets.find(w => w.id === tx.walletId)?.name}</p>
                                                        </div>
                                                    </div>
                                                    <div className="flex-grow mx-4 text-sm font-medium text-on-surface">
                                                        <div className="flex items-center gap-2">
                                                            {tx.transferId && <span title={t('transfer')}><ArrowsRightLeftIcon className="w-4 h-4 text-blue-500" /></span>}
                                                            {originalTx?.excludeFromReport && <span title={t('excludeFromReport')}><EyeSlashIcon className="w-4 h-4 text-gray-400" /></span>}
                                                            {tx.image && (
                                                                <button onClick={() => setSelectedImage(tx.image!)} className="flex-shrink-0 p-1 bg-primary/10 rounded-md hover:bg-primary/20 transition-colors" title="View Image">
                                                                    <PhotoIcon className="w-4 h-4 text-primary" />
                                                                </button>
                                                            )}
                                                            <span className="truncate">{tx.description}</span>
                                                        </div>
                                                    </div>
                                                    <div className="flex items-center justify-end gap-2 w-1/3">
                                                        <div className="text-right">
                                                            <p className={`font-semibold text-sm whitespace-nowrap ${tx.type === 'income' ? 'text-green-600' : 'text-red-600'}`}>
                                                                {tx.type === 'expense' ? '- ' : '+ '}{formatCurrency(tx.amount)}
                                                            </p>
                                                        </div>
                                                        <div className="flex ml-2 border-l pl-2">
                                                            <button onClick={() => handleEditClick(originalTx!)} className="text-primary p-1 hover:bg-primary/10 rounded-md transition-colors" title={t('editTransaction')} disabled={!!tx.transferId || tx.categoryId === 'cat-debt'}>
                                                                <PencilSquareIcon className="w-4 h-4"/>
                                                            </button>
                                                            <button onClick={() => handleDeleteClick(originalTx!)} className="text-red-600 p-1 hover:bg-red-50 rounded-md transition-colors ml-1" title={t('delete')}>
                                                                <TrashIcon className="w-4 h-4"/>
                                                            </button>
                                                        </div>
                                                    </div>
                                                </div>
                                            )
                                        })}
                                    </div>
                                </div>
                            );
                        })
                ) : (
                    <div className="bg-surface rounded-xl shadow-sm p-8 text-center text-on-surface-secondary border border-dashed">
                        <p>{t('noTransactionsMatchFilter')}</p>
                    </div>
                )}
            </div>
            <TransactionModal isOpen={isModalOpen} onClose={() => setIsModalOpen(false)} initialData={editingTransaction} />
        </div>
    );
};

export default Transactions;
