
import React, { useState, useEffect, FormEvent, useMemo, useRef } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Transaction, Category } from '../types';
import { XMarkIcon, Icon, CameraIcon, TrashIcon } from './icons/Icons';

interface TransactionModalProps {
    isOpen: boolean;
    onClose: () => void;
    initialData?: Transaction | null;
}

const TransactionModal: React.FC<TransactionModalProps> = ({ isOpen, onClose, initialData }) => {
    const { categories, wallets, addTransaction, updateTransaction, addDebtAndInitialTransaction, transactions } = useData();
    const { t, formatDate, parseDate, formatCurrencyInput, parseCurrencyInput } = useLanguage();
    const fileInputRef = useRef<HTMLInputElement>(null);
    
    const [mode, setMode] = useState<'income' | 'expense' | 'debt'>('expense');
    const [date, setDate] = useState(''); // YYYY-MM-DD
    const [displayDate, setDisplayDate] = useState(''); // DD/MM/YYYY
    const [description, setDescription] = useState('');
    const [amount, setAmount] = useState('');
    const [categoryId, setCategoryId] = useState('');
    const [walletId, setWalletId] = useState('');
    const [image, setImage] = useState<string | undefined>(undefined);
    const [error, setError] = useState('');

    // Debt state
    const [debtType, setDebtType] = useState<'loan' | 'debt'>('loan');
    const [person, setPerson] = useState('');
    
    // Share Bill state
    const [isShareBill, setIsShareBill] = useState(false);
    const [shareBillDone, setShareBillDone] = useState(false);
    const [isShareBillRepayment, setIsShareBillRepayment] = useState(false);
    const [linkedShareBillTxId, setLinkedShareBillTxId] = useState('');
    const [excludeFromReport, setExcludeFromReport] = useState(false);

    const handleShareBillRepaymentChange = (checked: boolean) => {
        setIsShareBillRepayment(checked);
        if (checked) {
            setExcludeFromReport(true);
        }
    };


    const sortedWallets = useMemo(() => {
        return [...wallets].sort((a, b) => a.order - b.order);
    }, [wallets]);

    useEffect(() => {
        if (isOpen) {
            const today = new Date().toISOString().split('T')[0];
            if (initialData) {
                setMode(initialData.type);
                setDate(initialData.date);
                setDisplayDate(formatDate(initialData.date));
                setDescription(initialData.description);
                setAmount(formatCurrencyInput(String(initialData.amount)));
                setCategoryId(initialData.categoryId);
                setWalletId(initialData.walletId);
                setImage(initialData.image);
                setPerson('');
                setDebtType('loan');
                setIsShareBill(initialData.isShareBill || false);
                setShareBillDone(initialData.shareBillDone || false);
                setIsShareBillRepayment(initialData.isShareBillRepayment || false);
                setLinkedShareBillTxId(initialData.linkedShareBillTxId || '');
                setExcludeFromReport(initialData.excludeFromReport || false);

            } else {
                setMode('expense');
                setDate(today);
                setDisplayDate(formatDate(today));
                setDescription('');
                setAmount('');
                setCategoryId('');
                setWalletId(sortedWallets.length > 0 ? sortedWallets[0].id : '');
                setImage(undefined);
                setError('');
                setPerson('');
                setDebtType('loan');
                setIsShareBill(false);
                setShareBillDone(false);
                setIsShareBillRepayment(false);
                setLinkedShareBillTxId('');
                setExcludeFromReport(false);
            }
        }
    }, [initialData, isOpen, sortedWallets, formatDate, formatCurrencyInput]);

    useEffect(() => {
        setCategoryId('');
        setPerson('');
        setIsShareBill(false);
        setIsShareBillRepayment(false);
        setLinkedShareBillTxId('');
    }, [mode]);

    const handleImageChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            const reader = new FileReader();
            reader.onloadend = () => {
                setImage(reader.result as string);
            };
            reader.readAsDataURL(file);
        }
    };

    const handleDateChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const newDisplayDate = e.target.value;
        setDisplayDate(newDisplayDate);
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            setDate(newIsoDate);
        }
    };

    const handleDateBlur = (e: React.FocusEvent<HTMLInputElement>) => {
        const newDisplayDate = e.target.value;
        const newIsoDate = parseDate(newDisplayDate);
        if (newIsoDate) {
            setDate(newIsoDate);
            setDisplayDate(formatDate(newIsoDate));
        } else if (newDisplayDate === '') {
            setDate('');
        } else {
            setDisplayDate(formatDate(date));
        }
    };

    const handleSubmit = (e: FormEvent) => {
        e.preventDefault();
        const numAmount = parseCurrencyInput(amount);

        if (mode === 'debt') {
            if (!person || !date || !walletId || numAmount <= 0) {
                setError(t('allFieldsRequired'));
                if (numAmount <= 0) setError(t('validAmount'));
                return;
            }
            addDebtAndInitialTransaction({
                type: debtType,
                person,
                amount: numAmount,
                date,
                walletId,
                description,
            });
        } else { // Income or Expense
            const categoryField = isShareBillRepayment ? linkedShareBillTxId : categoryId;
            if (!date || !description || !walletId || !categoryField || numAmount <= 0) {
                setError(t('allFieldsRequired'));
                 if (numAmount <= 0) setError(t('validAmount'));
                return;
            }

            const transactionData = {
                date,
                description,
                amount: numAmount,
                categoryId: isShareBillRepayment ? categories.find(c => c.name === 'Thu nhập khác')?.id || 'cat-i4' : categoryId,
                type: mode,
                walletId,
                image,
                isShareBill: mode === 'expense' && isShareBill,
                shareBillDone: mode === 'expense' && isShareBill && shareBillDone,
                isShareBillRepayment: mode === 'income' && isShareBillRepayment,
                linkedShareBillTxId: mode === 'income' && isShareBillRepayment ? linkedShareBillTxId : undefined,
                excludeFromReport: mode !== 'debt' ? excludeFromReport : false,
            };
            
            if (initialData) {
                updateTransaction({ ...transactionData, id: initialData.id });
            } else {
                addTransaction(transactionData);
            }
        }
        
        onClose();
    };

    const categoryOptions = useMemo(() => {
        const categoryMap = new Map<string, Category>(categories.map(c => [c.id, c]));
        
        const getCategoryType = (catId: string): 'income' | 'expense' | 'system' => {
             let current = categoryMap.get(catId);
             if (!current) return 'expense';

             if (['cat-transfer', 'cat-debt'].includes(current.id)) return 'system';
             
             while(current?.parentId) {
                 const parent = categoryMap.get(current.parentId);
                 if (!parent) return 'expense';
                 current = parent;
             }

             return current?.id === 'cat-i1' ? 'income' : 'expense';
        };

        const filteredCategories = categories.filter(c => {
            const type = getCategoryType(c.id);
            return type !== 'system' && type === mode;
        });

        interface CategoryTreeNode extends Category {
            children: CategoryTreeNode[];
        }

        const tree: CategoryTreeNode[] = [];
        const map = new Map<string, CategoryTreeNode>();
        filteredCategories.forEach(cat => map.set(cat.id, { ...cat, children: [] }));

        map.forEach(catNode => {
            if (catNode.parentId && map.has(catNode.parentId)) {
                map.get(catNode.parentId)!.children.push(catNode);
            } else {
                tree.push(catNode);
            }
        });

        const sortRecursive = (nodes: CategoryTreeNode[]) => {
            nodes.sort((a, b) => a.order - b.order);
            nodes.forEach(node => sortRecursive(node.children));
        };
        sortRecursive(tree);

        const flattenedOptions: React.ReactElement[] = [];
        const flatten = (nodes: CategoryTreeNode[], level: number) => {
            for (const node of nodes) {
                flattenedOptions.push(
                    <option key={node.id} value={node.id}>
                        {'\u00A0'.repeat(level * 4)}{node.name}
                    </option>
                );
                if (node.children.length > 0) {
                    flatten(node.children, level + 1);
                }
            }
        };

        flatten(tree, 0);
        return flattenedOptions;

    }, [categories, mode]);
    
    const openShareBills = useMemo(() => {
        return transactions.filter(tx => tx.isShareBill && !tx.shareBillDone);
    }, [transactions]);

    const selectedCategory = useMemo(() => categories.find(c => c.id === categoryId), [categoryId, categories]);

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex justify-center items-center z-50" aria-modal="true" role="dialog">
            <div className="bg-surface rounded-xl shadow-2xl w-full max-w-md m-4">
                <form onSubmit={handleSubmit}>
                    <div className="p-6 border-b border-gray-200 flex justify-between items-center">
                        <h2 className="text-xl font-bold">{initialData ? t('editTransaction') : t('addTransaction')}</h2>
                        <button type="button" onClick={onClose} className="text-on-surface-secondary hover:text-on-surface">
                            <XMarkIcon className="w-6 h-6" />
                        </button>
                    </div>

                    <div className="p-6 space-y-4 max-h-[70vh] overflow-y-auto">
                        <div className="grid grid-cols-3 gap-2 rounded-lg bg-gray-100 p-1">
                            <button type="button" onClick={() => setMode('expense')} className={`px-4 py-2 text-sm font-semibold rounded-md transition-colors ${mode === 'expense' ? 'bg-white shadow' : 'text-gray-600'}`}>
                                {t('expense')}
                            </button>
                            <button type="button" onClick={() => setMode('income')} className={`px-4 py-2 text-sm font-semibold rounded-md transition-colors ${mode === 'income' ? 'bg-white shadow' : 'text-gray-600'}`}>
                                {t('income')}
                            </button>
                            <button type="button" onClick={() => setMode('debt')} className={`px-4 py-2 text-sm font-semibold rounded-md transition-colors ${mode === 'debt' ? 'bg-white shadow' : 'text-gray-600'}`}>
                                {t('loanDebt')}
                            </button>
                        </div>
                        
                        {mode === 'expense' && (
                            <div className="flex items-center justify-between bg-gray-50 p-3 rounded-lg">
                                <label htmlFor="isShareBill" className="block text-sm font-medium text-on-surface-secondary">{t('shareBill')}</label>
                                <div className="flex items-center gap-4">
                                    {isShareBill && initialData && (
                                        <div className="flex items-center">
                                            <input type="checkbox" id="shareBillDone" checked={shareBillDone} onChange={e => setShareBillDone(e.target.checked)} className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary" />
                                            <label htmlFor="shareBillDone" className="ml-2 text-sm text-on-surface-secondary">{t('done')}</label>
                                        </div>
                                    )}
                                    <input type="checkbox" id="isShareBill" checked={isShareBill} onChange={e => setIsShareBill(e.target.checked)} className="h-5 w-5 rounded border-gray-300 text-primary focus:ring-primary" />
                                </div>
                            </div>
                        )}

                        {mode === 'income' && (
                            <div className="flex items-center bg-gray-50 p-3 rounded-lg">
                                <input type="checkbox" id="isShareBillRepayment" checked={isShareBillRepayment} onChange={e => handleShareBillRepaymentChange(e.target.checked)} className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary" />
                                <label htmlFor="isShareBillRepayment" className="ml-2 text-sm font-medium text-on-surface-secondary">{t('isShareBillRepayment')}</label>
                            </div>
                        )}

                        {mode === 'debt' && (
                            <div className="space-y-4 p-3 bg-gray-50 rounded-lg">
                                <div className="grid grid-cols-2 gap-2 rounded-lg bg-gray-200 p-1">
                                    <button type="button" onClick={() => setDebtType('loan')} className={`px-4 py-1 text-sm font-semibold rounded-md transition-colors ${debtType === 'loan' ? 'bg-white shadow' : 'text-gray-700'}`}>
                                        {t('loan')}
                                    </button>
                                    <button type="button" onClick={() => setDebtType('debt')} className={`px-4 py-1 text-sm font-semibold rounded-md transition-colors ${debtType === 'debt' ? 'bg-white shadow' : 'text-gray-700'}`}>
                                        {t('debt')}
                                    </button>
                                </div>
                                <div>
                                    <label htmlFor="person" className="block text-sm font-medium text-on-surface-secondary">{t('person')}</label>
                                    <input type="text" id="person" value={person} onChange={e => setPerson(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                                </div>
                            </div>
                        )}
                        
                        <div>
                            <label htmlFor="date" className="block text-sm font-medium text-on-surface-secondary">{t('date')}</label>
                            <input type="text" id="date" value={displayDate} onChange={handleDateChange} onBlur={handleDateBlur} placeholder="dd/mm/yyyy" className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                        </div>
                        
                        <div>
                            <label htmlFor="description" className="block text-sm font-medium text-on-surface-secondary">{t('description')} {mode !== 'debt' ? '' : <span className="text-xs">({t('optional')})</span>}</label>
                            <input type="text" id="description" value={description} onChange={e => setDescription(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                        </div>
                        
                         <div>
                            <label htmlFor="wallet" className="block text-sm font-medium text-on-surface-secondary">{t('wallet')}</label>
                            <select id="wallet" value={walletId} onChange={e => setWalletId(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                <option value="" disabled>{t('selectWallet')}</option>
                                {sortedWallets.map(w => <option key={w.id} value={w.id}>{w.name}</option>)}
                            </select>
                        </div>

                        <div className="grid grid-cols-2 gap-4">
                            <div>
                                <label htmlFor="amount" className="block text-sm font-medium text-on-surface-secondary">{t('amount')}</label>
                                <input type="text" inputMode="numeric" id="amount" value={amount} onChange={e => setAmount(formatCurrencyInput(e.target.value))} placeholder="0" className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface" />
                            </div>
                            {mode !== 'debt' && (
                                isShareBillRepayment ? (
                                    <div>
                                        <label htmlFor="linkedTx" className="block text-sm font-medium text-on-surface-secondary">{t('selectShareBillExpense')}</label>
                                        <select id="linkedTx" value={linkedShareBillTxId} onChange={e => setLinkedShareBillTxId(e.target.value)} className="mt-1 block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                            <option value="" disabled>{t('selectShareBillExpense')}</option>
                                            {openShareBills.map(tx => <option key={tx.id} value={tx.id}>{tx.date} - {tx.description}</option>)}
                                        </select>
                                    </div>
                                ) : (
                                <div>
                                    <label htmlFor="category" className="block text-sm font-medium text-on-surface-secondary">{t('category')}</label>
                                     <div className="flex items-center gap-2 mt-1">
                                        {selectedCategory && (
                                            <span className="w-7 h-7 rounded-full flex items-center justify-center flex-shrink-0" style={{ backgroundColor: selectedCategory.color + '20' }}>
                                                <Icon name={selectedCategory.icon} className="w-4 h-4" style={{ color: selectedCategory.color }} />
                                            </span>
                                        )}
                                        <select id="category" value={categoryId} onChange={e => setCategoryId(e.target.value)} className="block w-full border border-gray-300 rounded-md shadow-sm py-2 px-3 focus:outline-none focus:ring-primary focus:border-primary bg-white text-on-surface">
                                            <option value="" disabled>{t('selectCategory')}</option>
                                            {categoryOptions}
                                        </select>
                                    </div>
                                </div>
                                )
                            )}
                        </div>
                         
                        {/* Image Attachment Section */}
                        <div className="bg-gray-50 p-4 rounded-lg">
                            <label className="block text-sm font-medium text-on-surface-secondary mb-2">{t('addImage')}</label>
                            <div className="flex items-center gap-4">
                                {image ? (
                                    <div className="relative w-20 h-20 group">
                                        <img src={image} alt="Attachment" className="w-full h-full object-cover rounded-lg border shadow-sm" />
                                        <button 
                                            type="button" 
                                            onClick={() => setImage(undefined)} 
                                            className="absolute -top-2 -right-2 bg-red-500 text-white rounded-full p-1 shadow-md hover:bg-red-600 transition-colors"
                                        >
                                            <TrashIcon className="w-3 h-3" />
                                        </button>
                                    </div>
                                ) : (
                                    <button 
                                        type="button" 
                                        onClick={() => fileInputRef.current?.click()} 
                                        className="w-20 h-20 flex flex-col items-center justify-center border-2 border-dashed border-gray-300 rounded-lg hover:border-primary hover:bg-primary/5 transition-all text-gray-400 hover:text-primary"
                                    >
                                        <CameraIcon className="w-6 h-6 mb-1" />
                                        <span className="text-[10px] font-bold">{t('addImage')}</span>
                                    </button>
                                )}
                                <input 
                                    type="file" 
                                    ref={fileInputRef} 
                                    className="hidden" 
                                    accept="image/*" 
                                    onChange={handleImageChange} 
                                />
                            </div>
                        </div>

                        {mode !== 'debt' && (
                            <div className="flex items-center bg-gray-50 p-3 rounded-lg" title={t('excludeFromReportTooltip')}>
                                <input 
                                    type="checkbox" 
                                    id="excludeFromReport" 
                                    checked={excludeFromReport} 
                                    onChange={e => setExcludeFromReport(e.target.checked)} 
                                    className="h-4 w-4 rounded border-gray-300 text-primary focus:ring-primary" 
                                />
                                <label htmlFor="excludeFromReport" className="ml-2 text-sm font-medium text-on-surface-secondary">{t('excludeFromReport')}</label>
                            </div>
                        )}
                        
                        {error && <p className="text-sm text-red-600">{error}</p>}
                    </div>

                    <div className="p-6 bg-gray-50 rounded-b-xl flex justify-end items-center gap-4">
                        <button type="button" onClick={onClose} className="px-4 py-2 text-sm font-semibold text-on-surface-secondary bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                            {t('cancel')}
                        </button>
                        <button type="submit" className="px-4 py-2 text-sm font-semibold text-white bg-primary rounded-lg hover:bg-primary-dark transition-colors">
                            {t('save')}
                        </button>
                    </div>
                </form>
            </div>
        </div>
    );
};

export default TransactionModal;
