
import React, { useState, useMemo } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { Debt } from '../types';
import { PlusIcon, TrashIcon } from './icons/Icons';
import DebtModal from './DebtModal';
import RepaymentModal from './RepaymentModal';

const DebtItem: React.FC<{ debt: Debt }> = ({ debt }) => {
    const { t, language, formatDate } = useLanguage();
    const { deleteDebt } = useData();
    const [isRepaymentModalOpen, setIsRepaymentModalOpen] = useState(false);

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat(language === 'vi' ? 'vi-VN' : 'en-US', {
            style: 'currency', currency: language === 'vi' ? 'VND' : 'USD'
        }).format(amount);
    };

    const progress = debt.initialAmount > 0 ? ((debt.initialAmount - debt.outstandingAmount) / debt.initialAmount) * 100 : 100;

    const handleDelete = () => {
        if(window.confirm(t('confirmDeleteDebt'))) {
            deleteDebt(debt.id);
        }
    }

    return (
        <div className="bg-surface p-5 rounded-xl shadow-sm space-y-3">
            <div className="flex justify-between items-start">
                <div>
                    <h3 className="text-lg font-bold text-on-surface">{debt.person}</h3>
                    <p className="text-sm text-on-surface-secondary">{debt.description}</p>
                </div>
                <span className={`text-xs font-semibold px-2 py-1 rounded-full ${debt.status === 'paid' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'}`}>
                    {t(debt.status)}
                </span>
            </div>
            
            <div>
                <div className="flex justify-between text-sm mb-1">
                    <span className="text-on-surface-secondary">{formatCurrency(debt.initialAmount - debt.outstandingAmount)}</span>
                    <span className="font-semibold">{formatCurrency(debt.initialAmount)}</span>
                </div>
                <div className="w-full bg-gray-200 rounded-full h-2.5">
                    <div className="bg-primary h-2.5 rounded-full" style={{ width: `${progress}%` }}></div>
                </div>
            </div>

            <div className="flex justify-between items-center text-sm">
                <div className="text-on-surface-secondary">
                    {t('outstanding')}: <span className="font-bold text-red-600">{formatCurrency(debt.outstandingAmount)}</span>
                </div>
                 <div className="text-on-surface-secondary">
                    {debt.dueDate && <span>{t('dueDate')}: {formatDate(debt.dueDate)}</span>}
                </div>
            </div>
            
            <div className="border-t pt-3 flex justify-end gap-2">
                 <button onClick={handleDelete} className="p-1 text-red-500 hover:text-red-700">
                    <TrashIcon className="w-5 h-5"/>
                </button>
                <button 
                    onClick={() => setIsRepaymentModalOpen(true)}
                    disabled={debt.status === 'paid'}
                    className="text-sm bg-secondary text-white px-3 py-1.5 rounded-lg hover:bg-green-600 transition-colors disabled:bg-gray-400">
                    {t('recordRepayment')}
                </button>
            </div>
            
            <RepaymentModal 
                isOpen={isRepaymentModalOpen}
                onClose={() => setIsRepaymentModalOpen(false)}
                debt={debt}
            />
        </div>
    );
};

const DebtLedger: React.FC = () => {
    const { debts } = useData();
    const { t, language } = useLanguage();
    const [activeTab, setActiveTab] = useState<'loans' | 'debts'>('loans');
    const [isModalOpen, setIsModalOpen] = useState(false);

    const loans = debts.filter(d => d.type === 'loan');
    const myDebts = debts.filter(d => d.type === 'debt');

    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat(language === 'vi' ? 'vi-VN' : 'en-US', {
            style: 'currency', currency: language === 'vi' ? 'VND' : 'USD'
        }).format(amount);
    };

    const { totalLent, totalDebt } = useMemo(() => {
        return debts.reduce((acc, debt) => {
            if (debt.status === 'open') {
                if (debt.type === 'loan') {
                    acc.totalLent += debt.outstandingAmount;
                } else {
                    acc.totalDebt += debt.outstandingAmount;
                }
            }
            return acc;
        }, { totalLent: 0, totalDebt: 0 });
    }, [debts]);


    return (
        <div className="space-y-6">
            <div className="flex justify-between items-center">
                <h1 className="text-3xl font-bold text-on-surface">{t('debtLedger')}</h1>
                <button
                    onClick={() => setIsModalOpen(true)}
                    className="flex items-center gap-2 bg-primary text-white px-4 py-2 rounded-lg hover:bg-primary-dark transition-colors"
                >
                    <PlusIcon className="w-5 h-5" />
                    <span>{activeTab === 'loans' ? t('addLoan') : t('addDebt')}</span>
                </button>
            </div>
            
            {/* Stats Section */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                 <div className="bg-surface p-6 rounded-xl shadow-sm">
                    <h2 className="text-lg font-semibold text-on-surface-secondary">{t('totalLent')}</h2>
                    <p className="text-4xl font-bold mt-2 text-blue-600">
                        {formatCurrency(totalLent)}
                    </p>
                </div>
                 <div className="bg-surface p-6 rounded-xl shadow-sm">
                    <h2 className="text-lg font-semibold text-on-surface-secondary">{t('totalDebt')}</h2>
                    <p className="text-4xl font-bold mt-2 text-orange-600">
                        {formatCurrency(totalDebt)}
                    </p>
                </div>
            </div>

            <div>
                <div className="border-b border-gray-200">
                    <nav className="-mb-px flex space-x-6" aria-label="Tabs">
                        <button
                            onClick={() => setActiveTab('loans')}
                            className={`${activeTab === 'loans' ? 'border-primary text-primary' : 'border-transparent text-on-surface-secondary hover:text-on-surface hover:border-gray-300'} whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm transition-colors`}
                        >
                            {t('loans')}
                        </button>
                        <button
                            onClick={() => setActiveTab('debts')}
                            className={`${activeTab === 'debts' ? 'border-primary text-primary' : 'border-transparent text-on-surface-secondary hover:text-on-surface hover:border-gray-300'} whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm transition-colors`}
                        >
                            {t('debts')}
                        </button>
                    </nav>
                </div>
                
                <div className="pt-6">
                    {activeTab === 'loans' && (
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                            {loans.length > 0 ? loans.map(d => <DebtItem key={d.id} debt={d} />) : <p className="text-on-surface-secondary col-span-full">{t('noLoans')}</p>}
                        </div>
                    )}
                    {activeTab === 'debts' && (
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                            {myDebts.length > 0 ? myDebts.map(d => <DebtItem key={d.id} debt={d} />) : <p className="text-on-surface-secondary col-span-full">{t('noDebts')}</p>}
                        </div>
                    )}
                </div>
            </div>

            {/* FIX: Map activeTab state ('loans'/'debts') to the singular form ('loan'/'debt') expected by DebtModal's defaultType prop. */}
            <DebtModal
                isOpen={isModalOpen}
                onClose={() => setIsModalOpen(false)}
                defaultType={activeTab === 'loans' ? 'loan' : 'debt'}
            />
        </div>
    );
};

export default DebtLedger;
