
import React, { useMemo } from 'react';
import { useData } from '../context/DataContext';
import { useLanguage } from '../context/LanguageContext';
import { CreditCardIcon, BanknotesIcon, BuildingLibraryIcon, CircleStackIcon, Icon, EyeSlashIcon, SparklesIcon } from './icons/Icons';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer, Cell } from 'recharts';
import { Wallet } from '../types';
import Calendar from './Calendar';


const WalletTypeIcon = ({ type, className }: { type: Wallet['type'], className?: string }) => {
    switch (type) {
        case 'cash':
            return <BanknotesIcon className={className} />;
        case 'bank':
            return <BuildingLibraryIcon className={className} />;
        case 'credit':
            return <CreditCardIcon className={className} />;
        case 'crypto':
            return <CircleStackIcon className={className} />;
        default:
            return <CreditCardIcon className={className} />;
    }
};


const Dashboard: React.FC = () => {
    const { totalBalance, transactions, categories, wallets, getWalletBalance, netTransactionsForReporting, transactionsForDisplay } = useData();
    const { t, language, formatDate } = useLanguage();
    
    const formatCurrency = (amount: number) => {
        return new Intl.NumberFormat(language === 'vi' ? 'vi-VN' : 'en-US', { style: 'currency', currency: language === 'vi' ? 'VND' : 'USD' }).format(amount);
    }

    const lastUpdateDays = useMemo(() => {
        if (transactions.length === 0) return null;
        const latestTx = [...transactions].sort((a, b) => b.date.localeCompare(a.date))[0];
        const latestDate = new Date(`${latestTx.date}T00:00:00`);
        const today = new Date();
        today.setHours(0,0,0,0);
        const diffTime = Math.abs(today.getTime() - latestDate.getTime());
        return Math.floor(diffTime / (1000 * 60 * 60 * 24));
    }, [transactions]);

    const last12MonthsKeys = useMemo(() => {
        const keys = [];
        const now = new Date();
        for (let i = 11; i >= 0; i--) {
            const d = new Date(now.getFullYear(), now.getMonth() - i, 1);
            keys.push(`${d.getFullYear()}-${String(d.getMonth() + 1).padStart(2, '0')}`);
        }
        return keys;
    }, []);

    const electricityWaterData = useMemo(() => {
        const dataMap = new Map<string, { month: string, electric: number, water: number }>();
        
        last12MonthsKeys.forEach(key => {
            const [year, month] = key.split('-');
            const date = new Date(parseInt(year), parseInt(month) - 1, 1);
            const monthName = new Intl.DateTimeFormat(language === 'vi' ? 'vi-VN' : 'en-US', { 
                month: 'short',
                year: 'numeric'
            }).format(date);
            dataMap.set(key, { month: monthName, electric: 0, water: 0 });
        });

        transactions.forEach(t => {
            const tDate = new Date(`${t.date}T00:00:00`);
            const key = `${tDate.getFullYear()}-${String(tDate.getMonth() + 1).padStart(2, '0')}`;
            
            if (dataMap.has(key)) {
                const current = dataMap.get(key)!;
                const desc = t.description.toLowerCase();
                const isElectric = t.categoryId === 'cat-e13a' || desc.includes('tiền điện') || desc.includes('điện lực');
                const isWater = t.categoryId === 'cat-e13b' || desc.includes('tiền nước') || desc.includes('cấp nước');

                if (isElectric && t.type === 'expense') current.electric += t.amount;
                if (isWater && t.type === 'expense') current.water += t.amount;
            }
        });

        return Array.from(dataMap.values());
    }, [transactions, last12MonthsKeys, language]);

    const expenseByParentCategory = useMemo(() => {
        const data: { [key: string]: { amount: number, color: string, icon: string } } = {};
        
        netTransactionsForReporting
            .filter(t => t.type === 'expense' && t.categoryId !== 'cat-transfer')
            .forEach(t => {
                let currentCategory = categories.find(c => c.id === t.categoryId);
                let parentCategory = currentCategory;
                while(parentCategory && parentCategory.parentId) {
                    parentCategory = categories.find(c => c.id === parentCategory.parentId);
                }
                
                if (parentCategory) {
                    const parentName = parentCategory.name;
                    if (!data[parentName]) {
                        data[parentName] = { amount: 0, color: parentCategory.color, icon: parentCategory.icon };
                    }
                    data[parentName].amount += t.amount;
                }
            });
        
        const incomeParent = categories.find(c => c.id === 'cat-i1');
        if (incomeParent && data[incomeParent.name]) {
            delete data[incomeParent.name];
        }

        return Object.entries(data).map(([name, {amount, color}]) => ({ name, amount, color }));
    }, [netTransactionsForReporting, categories]);

    const monthlySummaryData = useMemo(() => {
        const months = new Map<string, { income: number; expense: number }>();
        last12MonthsKeys.forEach(key => months.set(key, { income: 0, expense: 0 }));

        netTransactionsForReporting.forEach(t => {
            const transactionDate = new Date(`${t.date}T00:00:00`);
            const monthKey = `${transactionDate.getFullYear()}-${String(transactionDate.getMonth() + 1).padStart(2, '0')}`;
            
            if (months.has(monthKey)) {
                const current = months.get(monthKey)!;
                if (t.type === 'income' && t.categoryId !== 'cat-transfer') {
                    current.income += t.amount;
                } else if (t.type === 'expense' && t.categoryId !== 'cat-transfer') {
                    current.expense += t.amount;
                }
            }
        });

        const locale = language === 'vi' ? 'vi-VN' : 'en-US';
        return Array.from(months.entries()).map(([key, data]) => {
            const [year, month] = key.split('-');
            const date = new Date(parseInt(year), parseInt(month) - 1, 1);
            const name = new Intl.DateTimeFormat(locale, { month: 'short' }).format(date);
            return { name, ...data };
        });
    }, [netTransactionsForReporting, language, last12MonthsKeys]);


    return (
        <div className="space-y-8">
            <h1 className="text-3xl font-bold text-on-surface">{t('dashboard')}</h1>

            {/* Reminder Alert */}
            {lastUpdateDays !== null && lastUpdateDays > 5 && (
                <div className="bg-orange-50 border-l-4 border-orange-500 p-4 rounded-r-lg shadow-sm flex items-center gap-3">
                    <SparklesIcon className="w-6 h-6 text-orange-500 flex-shrink-0" />
                    <p className="text-orange-800 font-medium">
                        {t('updateReminder', { days: lastUpdateDays })}
                    </p>
                </div>
            )}

            {/* Balances Section */}
             <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                <div className="lg:col-span-1 bg-surface p-6 rounded-xl shadow-sm border border-gray-100">
                    <h2 className="text-lg font-semibold text-on-surface-secondary">{t('totalBalance')}</h2>
                    <p className={`text-4xl font-bold mt-2 ${totalBalance >= 0 ? 'text-green-600' : 'text-red-600'}`}>
                        {formatCurrency(totalBalance)}
                    </p>
                </div>
                 <div className="lg:col-span-2 grid grid-cols-1 sm:grid-cols-2 gap-6">
                    {wallets.filter(w => w.showOnDashboard).map(wallet => (
                        <div key={wallet.id} className="bg-surface p-5 rounded-xl shadow-sm border border-gray-100">
                            <div className='flex items-center gap-3'>
                                <WalletTypeIcon type={wallet.type} className="w-6 h-6 text-primary"/>
                                <h3 className="text-lg font-semibold text-on-surface">{wallet.name}</h3>
                            </div>
                            <p className="text-2xl font-semibold mt-2 text-on-surface-secondary">
                                {formatCurrency(getWalletBalance(wallet.id))}
                            </p>
                        </div>
                    ))}
                 </div>
            </div>

            {/* Calendar Section */}
            <div className="bg-surface p-6 rounded-xl shadow-sm border border-gray-100">
                 <h2 className="text-xl font-bold mb-4">{t('financialCalendar')}</h2>
                 <Calendar />
            </div>

            {/* Electricity & Water Section */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                <div className="bg-surface p-6 rounded-xl shadow-sm border border-gray-100">
                    <div className="flex justify-between items-center mb-4">
                        <h2 className="text-xl font-bold flex items-center gap-2">
                            <span className="p-2 bg-yellow-100 rounded-lg"><Icon name="FireIcon" className="w-5 h-5 text-yellow-600" /></span>
                            {language === 'vi' ? 'Tiền điện (12 tháng)' : 'Electricity (12m)'}
                        </h2>
                    </div>
                    <div className="h-[280px]">
                        <ResponsiveContainer width="100%" height="100%">
                            <BarChart data={electricityWaterData}>
                                <CartesianGrid strokeDasharray="3 3" vertical={false} />
                                <XAxis dataKey="month" tick={{ fontSize: 10 }} />
                                <YAxis tickFormatter={(val) => new Intl.NumberFormat(language, { notation: 'compact' }).format(val)} tick={{ fontSize: 10 }} />
                                <Tooltip formatter={(val) => formatCurrency(val as number)} />
                                <Bar dataKey="electric" fill="#facc15" radius={[4, 4, 0, 0]} name={language === 'vi' ? 'Tiền điện' : 'Electricity'} />
                            </BarChart>
                        </ResponsiveContainer>
                    </div>
                </div>
                <div className="bg-surface p-6 rounded-xl shadow-sm border border-gray-100">
                    <div className="flex justify-between items-center mb-4">
                        <h2 className="text-xl font-bold flex items-center gap-2">
                            <span className="p-2 bg-blue-100 rounded-lg"><Icon name="BeakerIcon" className="w-5 h-5 text-blue-600" /></span>
                            {language === 'vi' ? 'Tiền nước (12 tháng)' : 'Water (12m)'}
                        </h2>
                    </div>
                    <div className="h-[280px]">
                        <ResponsiveContainer width="100%" height="100%">
                            <BarChart data={electricityWaterData}>
                                <CartesianGrid strokeDasharray="3 3" vertical={false} />
                                <XAxis dataKey="month" tick={{ fontSize: 10 }} />
                                <YAxis tickFormatter={(val) => new Intl.NumberFormat(language, { notation: 'compact' }).format(val)} tick={{ fontSize: 10 }} />
                                <Tooltip formatter={(val) => formatCurrency(val as number)} />
                                <Bar dataKey="water" fill="#38bdf8" radius={[4, 4, 0, 0]} name={language === 'vi' ? 'Tiền nước' : 'Water'} />
                            </BarChart>
                        </ResponsiveContainer>
                    </div>
                </div>
            </div>
           
            {/* Reports Section */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                <div className="bg-surface p-6 rounded-xl shadow-sm h-[350px] border border-gray-100">
                    <h2 className="text-xl font-bold mb-4">{t('spendingByTopLevelCategory')}</h2>
                    <ResponsiveContainer width="100%" height="90%">
                        <BarChart data={expenseByParentCategory} layout="vertical" margin={{ top: 5, right: 20, left: 20, bottom: 5 }}>
                            <CartesianGrid strokeDasharray="3 3" />
                            <XAxis type="number" tickFormatter={(value) => new Intl.NumberFormat(language).format(value as number)} />
                            <YAxis type="category" dataKey="name" tick={{ fontSize: 12 }} width={100} />
                            <Tooltip formatter={(value) => formatCurrency(value as number)} />
                            <Bar dataKey="amount" >
                                {expenseByParentCategory.map((entry, index) => (
                                    <Cell key={`cell-${index}`} fill={entry.color} />
                                ))}
                            </Bar>
                        </BarChart>
                    </ResponsiveContainer>
                </div>
                
                <div className="bg-surface p-6 rounded-xl shadow-sm h-[350px] border border-gray-100">
                    <h2 className="text-xl font-bold mb-4">{t('monthlySummary')}</h2>
                     <ResponsiveContainer width="100%" height="90%">
                        <BarChart data={monthlySummaryData} margin={{ top: 5, right: 20, left: -10, bottom: 5 }}>
                            <CartesianGrid strokeDasharray="3 3" />
                            <XAxis dataKey="name" tick={{ fontSize: 12 }} />
                            <YAxis tickFormatter={(value) => new Intl.NumberFormat(language, { notation: 'compact' }).format(value as number)} />
                            <Tooltip formatter={(value) => formatCurrency(value as number)} />
                            <Legend />
                            <Bar dataKey="income" fill="#22c55e" name={t('income')} />
                            <Bar dataKey="expense" fill="#ef4444" name={t('expense')} />
                        </BarChart>
                    </ResponsiveContainer>
                </div>
            </div>


            <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
                {/* Recent Transactions */}
                <div className="bg-surface p-6 rounded-xl shadow-sm border border-gray-100">
                    <h2 className="text-xl font-bold mb-4">{t('recentTransactions')}</h2>
                    <div className="space-y-4">
                        {transactionsForDisplay.slice(0,5).length > 0 ? (
                            transactionsForDisplay.slice(0,5).map(tx => {
                                const category = categories.find(c => c.id === tx.categoryId);
                                const originalTx = transactions.find(orig => orig.id === tx.id);
                                return (
                                <div key={tx.id} className="flex justify-between items-center">
                                    <div className='flex items-center gap-3'>
                                        {category && (
                                            <span className="w-9 h-9 rounded-full flex items-center justify-center flex-shrink-0" style={{ backgroundColor: category.color + '20' }}>
                                                <Icon name={category.icon} className="w-5 h-5" style={{ color: category.color }} />
                                            </span>
                                        )}
                                        <div>
                                            <div className="flex items-center gap-1.5">
                                                {originalTx?.excludeFromReport && <span title={t('excludeFromReport')}><EyeSlashIcon className="w-4 h-4 text-gray-400 flex-shrink-0" /></span>}
                                                <p className="font-semibold">{tx.description}</p>
                                            </div>
                                            <p className="text-sm text-on-surface-secondary">{formatDate(tx.date)} - {category?.name}</p>
                                        </div>
                                    </div>
                                    <p className={`font-semibold ${tx.type === 'income' ? 'text-green-500' : 'text-red-500'}`}>
                                        {tx.type === 'income' ? '+' : '-'} {formatCurrency(tx.amount)}
                                    </p>
                                </div>
                                )
                            })
                        ) : (
                            <p className="text-on-surface-secondary">{t('noTransactions')}</p>
                        )}
                    </div>
                </div>
            </div>
        </div>
    );
};

export default Dashboard;
